///////////////////////////////////////////////////////////////////////////
//
// File: model_photov.c
// Author: Michael Janssen
// Date: 1 Nov 2005
//
// CVS info:
//  $Source: /cvsroot/playerstage/code/stage/src/model_laser.c,v $
//  $Author: rtv $
//  $Revision: 1.76 $
//
///////////////////////////////////////////////////////////////////////////


#include <sys/time.h>
#include <math.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

#include "gui.h"

//#define DEBUG

#include "stage_internal.h"
extern stg_rtk_fig_t* fig_debug_rays;

#define TIMING 0

#define STG_PHOTOV_WATTS 1.5 // photov power consumption

#define STG_DEFAULT_PHOTOV_OFFSETX 0.0
#define STG_DEFAULT_PHOTOV_OFFSETY 0.0
#define STG_DEFAULT_PHOTOV_MAXOUT 255
#define STG_DEFAULT_PHOTOV_MINOUT 0

/**
@ingroup model
@defgroup model_photov PhotoVoltaic Model 
The photov model simulates a photovoltaic.

<h2>Worldfile properties</h2>

@par Summary and default values

@verbatim
photov
(
  # photov properties
  offsetx 0.0
  offsety 0.0
  minout 0
  maxout 255
	lightfile ""
)
@endverbatim

@par Details
- offsetx float
  - meters in the x direction (front/behind the robot) to sample the light image
- offsety float
  - meters in the y direction (left/right of robot) to sample the light image
- minout int
  - the minimum value reported by the sensor (black in lightfile)
- maxout int
  - the maximum value reported by the sensor (black in lightfile)
- lightfile string
  - the file to read the values from

**/

/** 
@ingroup stg_model_photov
@ingroup stg_model_props
@defgroup stg_model_photov_props Laser Properties

- "photov_cfg" stg_photov_config_t
- "photov_data" stg_photov_sample_t
**/

void photov_load( stg_model_t* mod )
{
  //if( wf_property_exists( mod->id, "

  stg_photov_config_t* now = 
    stg_model_get_property_fixed( mod, "photov_cfg", sizeof(stg_photov_config_t)); 

  stg_photov_config_t lconf;
  memset( &lconf, 0, sizeof(lconf));

  lconf.offsetx   = wf_read_length( mod->id, "offsetx", now ? now->offsetx : STG_DEFAULT_PHOTOV_OFFSETX );
  lconf.offsety   = wf_read_length( mod->id, "offsety", now ? now->offsety : STG_DEFAULT_PHOTOV_OFFSETY );
  lconf.minout    = wf_read_int( mod->id, "minout", now ? now->minout : STG_DEFAULT_PHOTOV_MINOUT );
  lconf.maxout    = wf_read_int( mod->id, "maxout", now ? now->maxout : STG_DEFAULT_PHOTOV_MAXOUT );
  lconf.lightfile = wf_read_string( mod->id, "lightfile",  now ? now->lightfile : "" );
  
	GError *pberror = NULL;
	GdkPixbuf *newpb = gdk_pixbuf_new_from_file(lconf.lightfile, &pberror);
	if (now->lightpb != NULL) {
		lconf.lightpb = now->lightpb; 
	}
	if (newpb != NULL) {
		lconf.lightpb = newpb; 
	} else {
		// TODO: Error here!
    PRINT_ERR2("couldn't open lightfile %s (%s)", lconf.lightfile, pberror->message);
	}
  stg_model_set_property( mod, "photov_cfg", &lconf, sizeof(lconf));
}

int photov_update( stg_model_t* mod );
int photov_startup( stg_model_t* mod );
int photov_shutdown( stg_model_t* mod );

int photov_render_data( stg_model_t* mod, char* name, 
		       void* data, size_t len, void* userp );
int photov_unrender_data( stg_model_t* mod, char* name,
			 void* data, size_t len, void* userp );
int photov_render_cfg( stg_model_t* mod, char* name, 
		      void* data, size_t len, void* userp );
//int laser_unrender_cfg( stg_model_t* mod, char* name,
//		void* data, size_t len, void* userp );


// not looking up color strings every redraw makes Stage 6% faster! (scanf is slow)
static int init = 0;
static stg_color_t photov_color=0, cfg_color=0, geom_color=0;

int photov_init( stg_model_t* mod )
{
  // we do this just the first time a laser is created
  if( init == 0 )
    {
      photov_color =   stg_lookup_color(STG_LASER_COLOR);
      geom_color = stg_lookup_color(STG_LASER_GEOM_COLOR);
      cfg_color = stg_lookup_color(STG_LASER_CFG_COLOR);
      init = 1;
    }

  // we don't consume any power until subscribed
  //mod->watts = 0.0; 
  
  // override the default methods
  mod->f_startup = photov_startup;
  mod->f_shutdown = photov_shutdown;
  mod->f_update =  NULL; // laser_update is installed startup, removed on shutdown
  mod->f_load = photov_load;

  // sensible laser defaults 
  stg_geom_t geom; 
  geom.pose.x = 0.0;
  geom.pose.y = 0.0;
  geom.pose.a = 0.0;
  geom.size.x = 0.0;
  geom.size.y = 0.0;
  stg_model_set_property( mod, "geom", &geom, sizeof(geom) );
      
  // remove the polygon: sensor has no body: 
  stg_model_set_property( mod, "polygons", NULL, 0);  

  // set up a photov-specific config structure
  stg_photov_config_t lconf;
  memset(&lconf,0,sizeof(lconf));  
  lconf.offsetx   = STG_DEFAULT_PHOTOV_OFFSETX;
  lconf.offsety   = STG_DEFAULT_PHOTOV_OFFSETY;
  lconf.maxout    = STG_DEFAULT_PHOTOV_MAXOUT;
  lconf.minout    = STG_DEFAULT_PHOTOV_MINOUT;
  stg_model_set_property( mod, "photov_cfg", &lconf, sizeof(lconf) );
  
  
  // set default color
  stg_model_set_property( mod, "color", &geom_color, sizeof(geom_color));
  
  // when data is set, render it to the GUI
  // clear the data - this will unrender it too
  // Start with blank data.
  stg_photov_sample_t sample; 
  memset(&sample, 0, sizeof(sample)); 

  stg_model_set_property( mod, "photov_data", &sample, sizeof(sample));

  // adds a menu item and associated on-and-off callbacks
  stg_model_add_property_toggles( mod, "photov_data", 
				  photov_render_data, // called when toggled on
				  NULL, 
				  photov_unrender_data, // called when toggled off
				  NULL, 
				  "photov data",
				  TRUE );

  // TODO
  /* stg_model_add_property_toggles( mod, "laser_cfg",  */
/* 				  laser_render_cfg, // called when toggled on */
/* 				  NULL, */
/* 				  NULL,//stg_fig_clear_cb, */
/* 				  NULL, //gui->cfg,  */
/* 				  "laser config", */
/* 				  TRUE ); */
  
  return 0;
}

// Returns 0 to 255 for black to white in lightmap 
int photov_get_pixel_for_globalxy( double x, double y, GdkPixbuf *pb, double maxx, double maxy ) 
{
	int imx = gdk_pixbuf_get_width(pb);
	int imy = gdk_pixbuf_get_height(pb);

	double normx = (maxx + x) / (maxx * 2.0); 
	double normy = (maxy - y) / (maxy * 2.0);
	
	int xim = floor(normx * imx); 
	int yim = floor(normy * imy);
	
	char *pixels = gdk_pixbuf_get_pixels(pb);
	
	unsigned char *pixelofinterest = pixels + gdk_pixbuf_get_rowstride(pb) * yim + gdk_pixbuf_get_n_channels(pb) * xim;

  PRINT_DEBUG4("Number of channels: %d, Pixel Value RGB: %d, %d, %d", gdk_pixbuf_get_n_channels(pb), pixelofinterest[0], pixelofinterest[1], pixelofinterest[2]);
	// Return the green channel.
	return pixelofinterest[1];
	
}

// TODO: This stuff
int photov_update( stg_model_t* mod )
{
  //puts( "laser update" );

  PRINT_DEBUG2( "[%lu] photov update (%d subs)", mod->world->sim_time, mod->subs );
  
  // no work to do if we're unsubscribed
  if( mod->subs < 1 )
    return 0;
  
  stg_photov_config_t* cfg = 
    stg_model_get_property_fixed( mod, "photov_cfg", sizeof(stg_photov_config_t));
  assert(cfg);

  stg_geom_t geom;
  stg_model_get_geom( mod, &geom );

  // get the sensed's pose in global coords
  stg_pose_t pz;
  memcpy( &pz, &geom.pose, sizeof(pz) );
	// Add on the offsets
	pz.x = pz.x + cfg->offsetx;
	pz.y = pz.y + cfg->offsety;
	// Convert to global
  stg_model_local_to_global( mod, &pz );

  PRINT_DEBUG2( "photov sampleat %.2f %.2f", pz.x, pz.y);

  stg_photov_sample_t sample;
	memset(&sample, 0, sizeof(sample));
	
	sample.value = photov_get_pixel_for_globalxy(pz.x, pz.y, cfg->lightpb, mod->world->width, mod->world->height);
	
	// Normalize
	sample.value = floor((((double)sample.value / 255.0) * (cfg->maxout - cfg->minout)) + cfg->minout);
	
	stg_model_set_property( mod, "photov_data", &sample, sizeof(sample));
  
  return 0; //ok
}


int photov_unrender_data( stg_model_t* mod, char* name, 
			 void* data, size_t len, void* userp )
{
  stg_model_fig_clear( mod, "photov_data_fig" );
  return 1; // callback just runs one time
}

int photov_render_data( stg_model_t* mod, char* name, 
		       void* data, size_t len, void* userp )
{
  
  stg_rtk_fig_t* fig = stg_model_get_fig( mod, "photov_data_fig" );  
  
  if( ! fig )
    fig = stg_model_fig_create( mod, "photov_data_fig", "top", STG_LAYER_LASERDATA );  
  
  stg_rtk_fig_clear( fig );

  
  stg_photov_config_t *cfg = 
    stg_model_get_property_fixed( mod, "photov_cfg", sizeof(stg_photov_config_t));
  assert( cfg );
  
  
  stg_photov_sample_t *sample = (stg_photov_sample_t*)data; 

	// TODO: Photov visualization
  
  return 0; // callback runs until removed
}

int photov_render_cfg( stg_model_t* mod, char* name, void* data, size_t len, void* userp )
{
  //puts( "laser render cfg" );
  
  stg_rtk_fig_t* fig = stg_model_get_fig( mod, "photov_cfg_fig" );
  
  if( !fig )
    fig = stg_model_fig_create( mod, "photov_config_fig", "top", STG_LAYER_LASERCONFIG );
  
  stg_rtk_fig_clear(fig);
  
  // TODO: photov config vis

  return 0;
}


/* int stg_model_unrender_cfg( stg_model_t* mod, char* name,  */
/* 			    void* data, size_t len, void* userp ) */
/* { */
/*   gui_model_t* gui =  */
/*     stg_model_get_property_fixed( mod, "gui", sizeof(gui_model_t)); */
  
/*   if( gui && gui->cfg  ) */
/*     stg_rtk_fig_clear(gui->cfg); */
  
/*   return 1; // callback just runs one time */
/* } */

int photov_startup( stg_model_t* mod )
{ 
  PRINT_DEBUG( "photov startup" );
  
  // start consuming power
  //mod->watts = STG_LASER_WATTS;
  
  // install the update function
  mod->f_update = photov_update;

  return 0; // ok
}

int photov_shutdown( stg_model_t* mod )
{ 
  PRINT_DEBUG( "photov shutdown" );
  
  // uninstall the update function
  mod->f_update = NULL;

  // stop consuming power
  //mod->watts = 0.0;
  
  // blank the data
  stg_photov_sample_t sample; 
  memset(&sample, 0, sizeof(sample)); 
  stg_model_set_property( mod, "photov_data", &sample, sizeof(sample) );

  return 0; // ok
}


