///////////////////////////////////////////////////////////////////////////
//
// File: model_hackbone.c
// Author: Michael Janssen
// Date: 29 Sep 2005
//
///////////////////////////////////////////////////////////////////////////

//#define DEBUG

#include <assert.h>
#include <math.h>
#include "stage_internal.h"
#include "gui.h"

const double STG_HACKBONE_WATTS = 0.0;

/** @defgroup model_hackbone Ranged Communication connectivity hack

<h2>Worldfile properties</h2>

@par Summary and default values

@verbatim
rangecom
(
  hackbone()
)
@endverbatim

*/

#define STG_DEFAULT_HACKBONE_MAX_HOPS 512

int hackbone_init ( stg_model_t* mod );
int hackbone_startup( stg_model_t* mod );
int hackbone_shutdown( stg_model_t* mod );
int hackbone_update( stg_model_t* mod );
void hackbone_load ( stg_model_t *mod );

int hackbone_render_data( stg_model_t* mod, char* name, void* data, size_t len, void* userp );
int hackbone_render_cfg( stg_model_t* mod, char* name, void* data, size_t len, void* userp );
int hackbone_unrender_data( stg_model_t* mod, char* name, void* data, size_t len, void* userp );
int hackbone_unrender_cfg( stg_model_t* mod, char* name, void* data, size_t len, void* userp );



int hackbone_init ( stg_model_t* mod ) {
  // override the default methods
  mod->f_startup = hackbone_startup;
  mod->f_shutdown = hackbone_shutdown;
  mod->f_update = NULL;
  mod->f_load = hackbone_load;
  PRINT_DEBUG("hackbone init"); 
  // remove the polygon: sensor has no body: 
  stg_model_set_property( mod, "polygons", NULL, 0); 

  stg_geom_t geom;
  memset( &geom, 0, sizeof(geom));
  stg_model_set_property( mod, "geom", &geom, sizeof(geom));  

  stg_color_t color = stg_lookup_color( "magenta" );
  stg_model_set_property( mod, "color", &color, sizeof(color) );
 
  stg_hackbone_config_t cfg;
  memset(&cfg, 0, sizeof(cfg));
  
  cfg.max_hops = STG_DEFAULT_HACKBONE_MAX_HOPS;
  cfg.rangecom = mod->parent;
  PRINT_DEBUG1("hackbone parent: %s", (cfg.rangecom)->token); 
 
  stg_model_set_property( mod, "hackbone_cfg", &cfg, sizeof(cfg)); 

  // start with no data
  stg_model_set_property( mod, "hackbone_data", NULL, 0 );

  // Menu is moved to here 
  stg_model_add_property_toggles( mod, "hackbone_data", 
      hackbone_render_data, 
      NULL, 
      hackbone_unrender_data,
      NULL, 
      "hackbone data", 
      TRUE ); 

  return 0;
}

void hackbone_load( stg_model_t* mod )
{
  stg_hackbone_config_t* now = 
    stg_model_get_property_fixed( mod, "hackbone_cfg", sizeof(stg_hackbone_config_t));
  assert(now); 

  stg_hackbone_config_t cfg;
  memset( &cfg, 0, sizeof(cfg) );
  
  cfg.max_hops = wf_read_length(mod->id, "max_hops", now->max_hops );
  cfg.rangecom = now->rangecom;
  
  stg_model_set_property(mod, "hackbone_cfg", &cfg, sizeof(cfg));
}



int hackbone_startup( stg_model_t* mod )
{
  PRINT_DEBUG( "hackbone startup" );  
  
  mod->f_update = hackbone_update;
  //mod->watts = STG_RANGECOM_WATTS;
  
  return 0;
}

int hackbone_shutdown( stg_model_t* mod )
{
  mod->f_update = NULL;
  //mod->watts = 0.0;

  // this will unrender the data
  stg_model_set_property( mod, "hackbone_data", NULL, 0 );
  
  return 0;
}

GArray *hackbone_build_connected_people(stg_model_t* mod, GArray *connected_hit, stg_model_t* firsthop, int hops) {

  // Have we already been checked? 
  for (int i = 0; i < connected_hit->len; i++) { 
    stg_hackbone_t x = g_array_index(connected_hit, stg_hackbone_t, i); 
    if (mod == x.model) { 
      if (x.hops < hops) {
        PRINT_DEBUG3("Visited %s in %d hops (%d hops now)...", mod->token, x.hops, hops); 
        return connected_hit;
      } else { 
        PRINT_DEBUG4("Removing %s at %d hops for %d hop path... (%d)", mod->token, x.hops, hops, i); 
        g_array_remove_index_fast(connected_hit, i); 
        i--;
      }
    } 
  } 

  stg_hackbone_t xnew; 
  memset(&xnew, 0, sizeof(stg_hackbone_t));

  xnew.model = mod; 
  if (hops < 2) { 
    firsthop = mod;
  }
  xnew.firsthop = firsthop; 
  xnew.hops = hops;

  PRINT_DEBUG2("Adding %s in %d hops...", (xnew.model)->token, xnew.hops); 
  g_array_append_val(connected_hit, xnew);

  size_t datalen = 0;
  stg_rangecom_t *data = stg_model_get_property( mod, "rangecom_data", &datalen);

  int bcount = datalen / sizeof(stg_rangecom_t); 
  
  //PRINT_DEBUG1("Model %s is connected..", mod->token);
  for (int i = 0; i < bcount; i++) { 
    hackbone_build_connected_people(data[i].model, connected_hit, firsthop, hops+1);
  } 
  //PRINT_DEBUG1("(Model %s done)", mod->token); 
  return connected_hit;
}

int hackbone_update( stg_model_t* mod )
{
  PRINT_DEBUG( "hackbone update" );

  if( mod->subs < 1 )
    return 0;
  
  stg_hackbone_config_t* cfg = 
    stg_model_get_property_fixed( mod, "hackbone_cfg", sizeof(stg_hackbone_config_t));

  //if( fig_debug_rays ) stg_rtk_fig_clear( fig_debug_rays );
  
  PRINT_DEBUG1( "HackBone Printing Tree From %s", (cfg->rangecom)->token ); 

  GArray *connected_hit = (GArray *) stg_model_get_property_fixed( mod, "hackbone_data", sizeof(GArray));
  if (connected_hit != NULL) { 
    PRINT_DEBUG1( "Killing old tree length %d old data.", connected_hit->len); 
    // g_array_free( connected_hit, FALSE );  XXX: MEMORY LEAK!
  }
  connected_hit = g_array_new( FALSE, TRUE, sizeof(stg_hackbone_t));
  hackbone_build_connected_people( cfg->rangecom, connected_hit, cfg->rangecom, 0); 
  printf("Connected to %s:\n", mod->token);
  for (int i = 0; i < connected_hit->len; i++) { 
    stg_hackbone_t x = g_array_index(connected_hit, stg_hackbone_t, i); 
    printf("-> %s through %s\n", (x.model)->token, (x.firsthop)->token); 
  } 
  PRINT_DEBUG1( "Setting hackbone_data size %d", connected_hit->len );
  stg_model_set_property( mod, "hackbone_data", connected_hit, sizeof(GArray));

  return 0;
}

int hackbone_unrender_data ( stg_model_t* mod, char* name, void* data, size_t len, void* userp ) {
  stg_model_fig_clear( mod, "hackbone_data_fig" ); 
  return 1; // cancel callback
} 

int hackbone_render_data( stg_model_t* mod, char* name, void* data, size_t len, void* userp )
{ 
  stg_rtk_fig_t* fig = stg_model_get_fig( mod, "hackbone_data_fig" );

  if (!fig) 
    fig = stg_model_fig_create( mod, "hackbone_data_fig", "top", STG_LAYER_NEIGHBORDATA ); 

  stg_rtk_fig_clear( fig ); 

  return 0;
}

int hackbone_render_cfg( stg_model_t* mod, char* name, void* data, size_t len, void* userp)
{
  return 0;
}

