/*
 *  Player - One Hell of a Robot Server
 *  Copyright (C) 2004, 2005 Richard Vaughan
 *                      
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
 * Desc: A plugin driver for Player that gives access to Stage devices.
 * Author: Richard Vaughan
 * Date: 10 December 2004
 * CVS: $Id: p_simulation.cc,v 1.12 2006/02/28 05:11:57 rtv Exp $
 */

// DOCUMENTATION ------------------------------------------------------------

/** @addtogroup player 
@par Simulation interface
- PLAYER_SIMULATION_REQ_SET_POSE2D
- PLAYER_SIMULATION_REQ_GET_POSE2D
*/

// CODE ------------------------------------------------------------

#define DEBUG

#include "p_driver.h"

// these are Player globals
extern bool player_quiet_startup;
extern PlayerTime* GlobalTime;

#define DRIVER_ERROR(X) printf( "Stage driver error: %s\n", X )

////////////////////////////////////////////////////////////////////////////////////

Interface::Interface(  player_devaddr_t addr, 
		       StgDriver* driver,
		       ConfigFile* cf, 
		       int section )
{
  this->last_publish_time = 0;
  this->addr = addr;
  this->driver = driver;
}

////////////////////////////////////////////////////////////////////////////////////

// player's cmdline args
//extern int global_argc;
//extern char** global_argv;

// 
// SIMULATION INTERFACE
//
InterfaceSimulation::InterfaceSimulation( player_devaddr_t addr, 
					  StgDriver* driver,
					  ConfigFile* cf, 
					  int section )
  : Interface( addr, driver, cf, section )
{
  printf( "a Stage world" ); fflush(stdout);
  //puts( "InterfaceSimulation constructor" );
  
  // boot libstage, requesting halt on any glib/gtk/gnome problem
//   int argc = 2;
//   char* argv[2];
//   argv[0] = "player";
//   argv[1] = "--g-fatal-warnings";
//   stg_init( argc, argv );

  const char* worldfile_name = cf->ReadString(section, "worldfile", NULL );
  
  if( worldfile_name == NULL )
    {
      PRINT_ERR1( "device \"%s\" uses the Stage driver but has "
		  "no \"model\" value defined. You must specify a "
		  "model name that matches one of the models in "
		  "the worldfile.",
		  worldfile_name );
      return; // error
    }
  
  char fullname[MAXPATHLEN];
  
  if( worldfile_name[0] == '/' )
    strcpy( fullname, worldfile_name );
  else
    {
      char *tmp = strdup(cf->filename);
      snprintf( fullname, MAXPATHLEN, 
		"%s/%s", dirname(tmp), worldfile_name );      
      free(tmp);
    }
  
  // a little sanity testing
  if( !g_file_test( fullname, G_FILE_TEST_EXISTS ) )
    {
      PRINT_ERR1( "worldfile \"%s\" does not exist", worldfile_name );
      return;
    }
  
  // create a passel of Stage models in the local cache based on the
  // worldfile
  
  StgDriver::world = stg_world_create_from_file( fullname );
  assert(StgDriver::world);
  //printf( " done.\n" );
  
  // steal the global clock - a bit aggressive, but a simple approach
  if( GlobalTime ) delete GlobalTime;
  assert( (GlobalTime = new StgTime( driver ) ));
  
  // start the simulation
  // printf( "  Starting world clock... " ); fflush(stdout);
  //stg_world_resume( world );
  
  StgDriver::world->paused = FALSE;
  
  // this causes Driver::Update() to be called even when the device is
  // not subscribed
  driver->alwayson = TRUE;    
  
  // Start the device thread; spawns a new thread and executes
  // StgDriver::Main(), which contains the main loop for the driver.
  //puts( "\nStarting thread" );
  //driver->StartThread();

  puts( "" ); // end the Stage startup line
}      

int InterfaceSimulation::ProcessMessage(MessageQueue* resp_queue,
                                        player_msghdr_t* hdr,
                                        void* data)
{
  // Is it a request to set a model's pose?
  if(Message::MatchMessage(hdr, PLAYER_MSGTYPE_REQ, 
                           PLAYER_SIMULATION_REQ_SET_POSE2D, 
                           this->addr))
  {
    player_simulation_pose2d_req_t* req = 
            (player_simulation_pose2d_req_t*)data;

    stg_pose_t pose;
    pose.x = req->pose.px;
    pose.y = req->pose.py;
    pose.a = req->pose.pa;

    printf( "Stage: received request to move object \"%s\" to (%.2f,%.2f,%.2f)\n",
            req->name, pose.x, pose.y, pose.a );

    // look up the named model

    stg_model_t* mod = 
            stg_world_model_name_lookup( StgDriver::world, req->name );

    if( mod )
    {
      // move it 
      printf( "moving model \"%s\"\n", req->name );	    
      stg_model_set_pose( mod, &pose );

      this->driver->Publish(this->addr, resp_queue,
                            PLAYER_MSGTYPE_RESP_ACK,
                            PLAYER_SIMULATION_REQ_SET_POSE2D);
      return(0);
    }
    else
    {
      PRINT_WARN1( "SETPOSE2D request: simulation model \"%s\" not found", req->name );
      return(-1);
    }
  }
  // Is it a request to get a model's pose?
  else if(Message::MatchMessage(hdr, PLAYER_MSGTYPE_REQ, 
                                PLAYER_SIMULATION_REQ_GET_POSE2D, 
                                this->addr))
  {
    player_simulation_pose2d_req_t* req = 
            (player_simulation_pose2d_req_t*)data;

    printf( "Stage: received request for position of object \"%s\"\n", req->name );

    // look up the named model	
    stg_model_t* mod = 
            stg_world_model_name_lookup( StgDriver::world, req->name );

    if( mod )
    {
      stg_pose_t* pose = &mod->pose;

      printf( "Stage: returning location (%.2f,%.2f,%.2f)\n",
              pose->x, pose->y, pose->a );


      player_simulation_pose2d_req_t reply;
      memcpy( &reply, req, sizeof(reply));
      reply.pose.px = pose->x;
      reply.pose.py = pose->y;
      reply.pose.pa = pose->a;

      /*
         printf( "Stage: returning location (%d %d %d)\n",
         reply.x, reply.y, reply.a );
       */

      this->driver->Publish( this->addr, resp_queue, 
                             PLAYER_MSGTYPE_RESP_ACK, 
                             PLAYER_SIMULATION_REQ_GET_POSE2D,
                             (void*)&reply, sizeof(reply), NULL );
      return(0);
    }
    else
    {
      PRINT_WARN1( "Stage: GETPOSE2D request: simulation model \"%s\" not found", req->name );
      return(-1);
    }
  }
  else
  {
    // Don't know how to handle this message.
    PRINT_WARN2( "stg_simulation doesn't support msg with type/subtype %d/%d",
		 hdr->type, hdr->subtype);
    return(-1);
  }
}
