///////////////////////////////////////////////////////////////////////////
//
// File: model_rangecom.c
// Author: Michael Janssen
// Date: 19 May 2005
//
// Based on model_fiducial.c by Richard Vaughan
//
// CVS info:
//  $Source: /cvsroot/playerstage/code/stage/src/model_fiducial.c,v $
//  $Author: rtv $
//  $Revision: 1.37 $
//
///////////////////////////////////////////////////////////////////////////

//#define DEBUG

#include <assert.h>
#include <math.h>
#include "stage_internal.h"
#include "gui.h"

extern stg_rtk_fig_t* fig_debug_rays;

#define STG_DEFAULT_RANGECOM_RANGEMAX 8
#define STG_DEFAULT_RANGECOM_WALLPENALTY 2

#define STG_RANGECOM_MAX 64

const double STG_RANGECOM_WATTS = 10.0;

/** @defgroup model_rangecom Ranged Communication detector model
The model simulates a ranged communication link (such as 802.11 or bluetooth)

<h2>Worldfile properties</h2>

@par Summary and default values

@verbatim
rangecom
(
  # rangecom properties
  range_max 8.0
  wall_penalty 0.5


  # model properties
  size [0 0]
)
@endverbatim

@par Details
- range_max float
  - the maximum range at which the connection is available.   This range will by def
- range_max_id float
  - the maximum range at which the sensor can detect the ID of a fiducial, in meters.
- fov float
  - the angular field of view of the scanner, in degrees. 

*/

int rangecom_init ( stg_model_t* mod );
int rangecom_startup( stg_model_t* mod );
int rangecom_shutdown( stg_model_t* mod );
int rangecom_update( stg_model_t* mod );
void rangecom_load ( stg_model_t *mod );

int rangecom_render_data( stg_model_t* mod, char* name, void* data, size_t len, void* userp );
int rangecom_render_cfg( stg_model_t* mod, char* name, void* data, size_t len, void* userp );
int rangecom_unrender_data( stg_model_t* mod, char* name, void* data, size_t len, void* userp );
int rangecom_unrender_cfg( stg_model_t* mod, char* name, void* data, size_t len, void* userp );



int rangecom_init ( stg_model_t* mod ) {
  // override the default methods
  mod->f_startup = rangecom_startup;
  mod->f_shutdown = rangecom_shutdown;
  mod->f_update = NULL; // installed at startup/shutdown
  mod->f_load = rangecom_load;

  // remove the polygon: sensor has no body: 
  stg_model_set_property( mod, "polygons", NULL, 0); 

  stg_geom_t geom;
  memset( &geom, 0, sizeof(geom));
  stg_model_set_property( mod, "geom", &geom, sizeof(geom));  

  stg_rangecom_return_t rcr = RangecomPoint;
  stg_model_set_property( mod, "rangecom_return", &rcr, sizeof(rcr));

  stg_color_t color = stg_lookup_color( "magenta" );
  stg_model_set_property( mod, "color", &color, sizeof(color) );
 
  stg_rangecom_config_t cfg;
  memset(&cfg, 0, sizeof(cfg));
  
  cfg.max_range = STG_DEFAULT_RANGECOM_RANGEMAX;
  cfg.wall_penalty = STG_DEFAULT_RANGECOM_WALLPENALTY;
 
  stg_model_set_property( mod, "rangecom_cfg", &cfg, sizeof(cfg)); 

  // start with no data
  stg_model_set_property( mod, "rangecom_data", NULL, 0 );

  // Menu is moved to here? 
  stg_model_add_property_toggles( mod, "rangecom_data", 
      rangecom_render_data, 
      NULL, 
      rangecom_unrender_data,
      NULL, 
      "rangecom data", 
      TRUE ); 

  return 0;
}

void rangecom_load( stg_model_t* mod )
{
  stg_rangecom_config_t* now = 
    stg_model_get_property_fixed( mod, "rangecom_cfg", sizeof(stg_rangecom_config_t));
  assert(now); 

  stg_rangecom_config_t cfg;
  memset( &cfg, 0, sizeof(cfg) );
  
  cfg.max_range = wf_read_length(mod->id, "range_max", now->max_range );
  cfg.wall_penalty = wf_read_length(mod->id, "wall_penalty", now->wall_penalty);
  
  stg_model_set_property(mod, "rangecom_cfg", &cfg, sizeof(cfg));
}



int rangecom_startup( stg_model_t* mod )
{
  PRINT_DEBUG( "rangecom startup" );  
  
  mod->f_update = rangecom_update;
  //mod->watts = STG_RANGECOM_WATTS;
  
  return 0;
}

int rangecom_shutdown( stg_model_t* mod )
{
  mod->f_update = NULL;
  //mod->watts = 0.0;

  // this will unrender the data
  stg_model_set_property( mod, "rangecom_data", NULL, 0 );
  
  return 0;
}

// TODO: Find out if model_rangecom_buffer_t (based on model_fiducial_buffer_t) has redundant stuff (it probably does)
typedef struct 
{
  stg_model_t* mod;
  stg_pose_t pose;
  stg_rangecom_config_t cfg;
  GArray* peers;
} model_rangecom_buffer_t;


// Used as a matching function for raytrace.  With current stage, should be called 
//   ONCE for each object on the raytrace. 
int rangecom_raytrace_match( stg_model_t* target, stg_model_t* hitmod )
{
  stg_rangecom_return_t* his_rcr = 
    stg_model_get_property_fixed( target, "rangecom_return", sizeof(stg_rangecom_return_t));
  return his_rcr != 0; 
}	


void model_rangecom_check_neighbor( gpointer key, gpointer value, gpointer user )
{
  model_rangecom_buffer_t* mfb = (model_rangecom_buffer_t*)user;
  stg_model_t* him = (stg_model_t*)value;
  
		
  // don't compare a model with itself, and don't consider models which 
  // are not rangecom devices
  
  stg_rangecom_return_t* his_rcr = 
    stg_model_get_property_fixed( him, "rangecom_return", sizeof(stg_rangecom_return_t));
  if( his_rcr == NULL || him == mfb->mod || (*his_rcr) != RangecomPoint ) 
    return; 
  
	PRINT_DEBUG2( "checking model %s - he has rangecom value %d",
  	him->token, rcr );

  stg_pose_t hispose;
  stg_model_get_global_pose( him, &hispose );
  
  double dx = hispose.x - mfb->pose.x;
  double dy = hispose.y - mfb->pose.y;
  
  // are we within range?
  double range = hypot( dy, dx );
  if( range > mfb->cfg.max_range ) {
    PRINT_DEBUG1( "but model %s is outside my range", him->token);
    return;
  }

    
  // is he in my field of view?
  double hisbearing = atan2( dy, dx );
  /* mjanssen - we don't have a field of view, we can see in 360
  double dif = mfb->pose.a - hisbearing;

  if( fabs(NORMALIZE(dif)) > mfb->cfg.fov/2.0 )
    {
      //PRINT_DEBUG1( "  but model %s is outside my FOV", him->token);
      return;
    }
  */  
  // TODO: Make this not line-of-sight but count the walls..
  stg_meters_t impedencedist = 0;
  stg_pose_t frompose = mfb->pose;
  itl_t* itl = NULL; 
  itl = itl_create( frompose.x, frompose.y, hispose.x, hispose.y, 
                    him->world->matrix, PointToPoint );
  PRINT_DEBUG1("Ray originating from model %s", mfb->mod->token); 
  
  impedencedist = itl_measure_distance_through_matching( itl, rangecom_raytrace_match, mfb->mod );
  itl_destroy( itl );
 
  // TODO: Discount range based on the amount of walls we hit. 

  PRINT_DEBUG1( "finished raytracing, went through %f meters of wall", impedencedist );
  
  // if it was him and he's in range, we can see him
	if ( (mfb->cfg.max_range - (mfb->cfg.wall_penalty*impedencedist)) > range ) { 
		// mjanssen - we don't get his geometry. 
		/*
		stg_geom_t hisgeom;
		stg_model_get_geom(him,&hisgeom);
		*/

		// record where we saw him and what he looked like
		stg_rangecom_t peer;      
		peer.range = range;
		peer.bearing = NORMALIZE( hisbearing - mfb->pose.a);
		// TODO: Make this more useful - rangecom_name possibly? 
		strncpy(peer.name, him->token, 31);
		
		// if he's within ID range, get his fiducial.return value, else
		// we see value 0
		// mjanssen - XXX: make this take down other robot's IP or something
		// fid.id = range < mfb->cfg.max_range_id ? him->fiducial_return : 0;

		//PRINT_DEBUG2( "adding %s's value %d to my list of fiducials",
		//	    him->token, him->fiducial_return );

		g_array_append_val( mfb->peers, peer );
	}
}


///////////////////////////////////////////////////////////////////////////
// Update the beacon data
// TODO: fix model_fiducial_buffer_t to model_rangecom_buffer_t
int rangecom_update( stg_model_t* mod )
{
  PRINT_DEBUG( "rangecom update" );

  if( mod->subs < 1 )
    return 0;

  if( fig_debug_rays ) stg_rtk_fig_clear( fig_debug_rays );
  
  model_rangecom_buffer_t mrb;
  memset( &mrb, 0, sizeof(mrb) );
  
  mrb.mod = mod;
  
  stg_rangecom_config_t* cfg = 
    stg_model_get_property_fixed( mod, "rangecom_cfg", sizeof(stg_rangecom_config_t) );
  assert(cfg);

  memcpy(&mrb.cfg, cfg, sizeof(mrb.cfg)); 

  mrb.peers = g_array_new( FALSE, TRUE, sizeof(stg_rangecom_t) );
  stg_model_get_global_pose( mod, &mrb.pose );
  
  // for all the objects in the world
  g_hash_table_foreach( mod->world->models, model_rangecom_check_neighbor, &mrb );

  //PRINT_DEBUG2( "model %s saw %d fiducials", mod->token, mfb.fiducials->len );

  stg_model_set_property( mod, "rangecom_data",
		  mrb.peers->data, 
		  mrb.peers->len * sizeof(stg_rangecom_t) );
  
  g_array_free( mrb.peers, TRUE );

  return 0;
}

int rangecom_unrender_data ( stg_model_t* mod, char* name, void* data, size_t len, void* userp ) {
  stg_model_fig_clear( mod, "rangecom_data_fig" ); 
  return 1; // cancel callback
} 

int rangecom_render_data( stg_model_t* mod, char* name, void* data, size_t len, void* userp )
{ 
  stg_rtk_fig_t* fig = stg_model_get_fig( mod, "rangecom_data_fig" );

  if (!fig) 
    fig = stg_model_fig_create( mod, "rangecom_data_fig", "top", STG_LAYER_NEIGHBORDATA ); 

  stg_rtk_fig_clear( fig ); 

  stg_rtk_fig_color_rgb32( fig, stg_lookup_color( STG_RANGECOM_COLOR ) );
  
  stg_rangecom_t *ranges = (stg_rangecom_t*) data; 

  int bcount = len / sizeof(stg_rangecom_t);
  
  ///char text[32];

  int b;
  for( b=0; b < bcount; b++ )
    {
      // TODO: RENDER RANGECOM RETURNS
      // the location of the target
      double pa = ranges[b].bearing;
      double px = ranges[b].range * cos(pa); 
      double py = ranges[b].range * sin(pa);
      
      stg_rtk_fig_line( fig, 0, 0, px, py );	
      
    }

  return 0;
}

int rangecom_render_cfg( stg_model_t* mod, char* name, void* data, size_t len, void* userp)
{
  stg_rangecom_config_t *cfg = data; 
  assert(cfg); 

  stg_rtk_fig_t* fig = stg_model_get_fig(mod, "rangecom_cfg_fig"); 

  if (!fig)
    stg_model_fig_create(mod, "rangecom_cfg_fig", "top", STG_LAYER_NEIGHBORCONFIG); 
      
  stg_rtk_fig_color_rgb32( fig, stg_lookup_color( STG_RANGECOM_CFG_COLOR ));
  
	// 360 degrees 
	stg_point_t* points = calloc( sizeof(stg_point_t), 360);
  stg_pose_t frompose; 
	stg_model_get_global_pose(mod, &frompose); 

  stg_geom_t geom; 
	stg_model_get_geom( mod, &geom); 
	double bearing = frompose.a; 
	double sample_incr = (2.0 * M_PI) / 360.0;
	double impedence = 0;
	double actualrange = 0;
	itl_t* itl = NULL;
	for (int s = 0; s < 360; s++) { 
		// range for this degree
		itl = itl_create( frompose.x, frompose.y, bearing, cfg->max_range,
					                    mod->world->matrix, PointToBearingRange );
		impedence = itl_measure_distance_through_matching( itl, rangecom_raytrace_match, mod );
		itl_destroy( itl );
		actualrange = cfg->max_range - (cfg->wall_penalty * impedence); 
		PRINT_DEBUG3("cfg: bearing: %d imp: %f, actualrange: %f", s, impedence, actualrange); 

		points[s].x = actualrange * cos(bearing - frompose.a); 
		points[s].y = actualrange * sin(bearing - frompose.a); 
		bearing += sample_incr;
	}

	stg_rtk_fig_polygon( fig, 0, 0, 0, 360, points, FALSE ); 

	free (points); 
}

