///////////////////////////////////////////////////////////////////////////
//
// File: model_gps.c
// Author: Michael Janssen
// Date: 20 July 2005
//
///////////////////////////////////////////////////////////////////////////

#define DEBUG

#include <assert.h>
#include <math.h>
#include "stage_internal.h"
#include "gui.h"

extern stg_rtk_fig_t* fig_debug_rays;

const double STG_GPS_WATTS = 10.0;

#define STG_DEFAULT_GPS_LATBASE 0.0 
#define STG_DEFAULT_GPS_LONBASE 0.0 


/** @defgroup model_gps GPS model
The model simulates a GPS with optional sources of error

<h2>Worldfile properties</h2>

@par Summary and default values

@verbatim
gps
(
  # gps properties
  lat_base 0.0
  lon_base 0.0

  # model properties
  size [0 0]
)
@endverbatim

@par Details
- lat_base float
  - the latitude base, in degrees, default 0
- lon_base float
  - the longitude base, in degrees, default 0

*/

void gps_load( stg_model_t* mod )
{
  stg_gps_config_t* now = 
    stg_model_get_property_fixed(mod, "gps_cfg", sizeof(stg_gps_config_t)); 

  assert(now); 

  stg_gps_config_t cfg;
  memset( &cfg, 0, sizeof(cfg) );
  
  cfg.lat_base = wf_read_float(mod->id, "lat_base", now->lat_base );
  cfg.lon_base = wf_read_float(mod->id, "lon_base", now->lon_base );
  
  stg_model_set_property(mod, "gps_cfg", &cfg, sizeof(stg_gps_config_t)); 
}

int gps_startup( stg_model_t* mod );
int gps_shutdown( stg_model_t* mod );
int gps_update( stg_model_t* mod );
int gps_render_data( stg_model_t* mod, char* name, void* data, size_t len, void* userp );
int gps_render_cfg( stg_model_t* mod, char* name, void* data, size_t len, void* userp );
int gps_unrender_data( stg_model_t* mod, char* name, void* data, size_t len, void* userp );

int gps_init( stg_model_t* mod ) {
  // override the default methods
  mod->f_startup = gps_startup;
  mod->f_shutdown = gps_shutdown;
  mod->f_update = NULL; // installed at startup/shutdown
  mod->f_load = gps_load;

  // no polygons 
  stg_model_set_property( mod, "polygons", NULL, 0); 

  stg_geom_t geom;
  memset( &geom, 0, sizeof(geom));
  stg_model_set_property( mod, "geom", &geom , sizeof(geom) );  

  stg_color_t color = stg_lookup_color( "magenta" );
  stg_model_set_property( mod, "color", &color, sizeof(color) );

	stg_gps_t data;
	memset( &data, 0, sizeof(data));

  // model is not dataless
	stg_model_set_property(mod, "gps_data", &data, sizeof(stg_gps_t)); 

  // default parameters
  stg_gps_config_t cfg; 
  cfg.lon_base = STG_DEFAULT_GPS_LONBASE;
  cfg.lat_base = STG_DEFAULT_GPS_LATBASE;
  stg_model_set_property( mod, "gps_cfg", &cfg, sizeof(cfg) );

  // maybe add a data display later in life
  
  return 0;
}

int gps_startup( stg_model_t* mod )
{
  PRINT_DEBUG( "gps startup" );  
  
  mod->f_update = gps_update;
  //mod->watts = STG_GPS_WATTS;
  
  return 0;
}

int gps_shutdown( stg_model_t* mod )
{
  mod->f_update = NULL;
  //mod->watts = 0.0;

  // this will unrender the data
  stg_model_set_property ( mod, "gps_data", NULL, 0 );
  
  return 0;
}

///////////////////////////////////////////////////////////////////////////
// GPS Update!
int gps_update( stg_model_t* mod )
{
  PRINT_DEBUG( "gps update" );

  if( mod->subs < 1 )
    return 0;

	stg_pose_t pos; 

	stg_model_get_global_pose(mod, &pos);

	stg_gps_config_t *cfg =
     stg_model_get_property_fixed( mod, "gps_cfg", sizeof(stg_gps_config_t) ); 

  stg_gps_t data;

  struct timeval curtime;
  gettimeofday(&curtime, NULL);

  data.sec = curtime.tv_sec;
  data.usec = curtime.tv_usec;
	// NOTE: Units are deg 
  data.latitude =  (pos.x  / (2.0 * M_PI * 6356752.3142 / 360.0)) + cfg->lat_base;
  data.longitude = (pos.y / (2.0 * M_PI * 6378137 / 360.0)) + cfg->lon_base;

	PRINT_DEBUG4("(%f, %f) -> gps lat: %f lon: %f", pos.x, pos.y, data.latitude, data.longitude); 
	stg_model_set_property(mod, "gps_data", &data, sizeof(stg_gps_t)); 

  return 0;
}

int gps_render_data( stg_model_t* mod, char* name, void* data, size_t len, void* userp )
{ 
  return 0; 
}

int gps_unrender_data( stg_model_t* mod, char* name, void* data, size_t len, void* userp)
{
  return 1;
}

int gps_render_cfg( stg_model_t* mod, char* name, void* data, size_t len, void* userp )
{ 
  return 0;
}

