///////////////////////////////////////////////////////////////////////////
//
// File: model_compass.c
// Author: Michael Janssen
// Date: 2 Aug 2005
//
// CVS info:
//  $Source: /cvsroot/playerstage/code/stage/src/model_fiducial.c,v $
//  $Author: rtv $
//  $Revision: 1.37 $
//
///////////////////////////////////////////////////////////////////////////

//#define DEBUG

#include <assert.h>
#include <math.h>
#include "stage_internal.h"
#include "gui.h"

#define STG_DEFAULT_COMPASS_MAGNORTH 0.0

const double STG_COMPASS_WATTS = 0.0;

/** @defgroup model_compass Compass model
The model simulates a compass which points to magnetic north.

<h2>Worldfile properties</h2>

@par Summary and default values

@verbatim
compass
(
  # compass properties
  north 0.0 

  # model properties
  size [0 0]
)
@endverbatim

@par Details
- north float
  - the degree in the global pose which is magnetic north.

*/

int compass_init ( stg_model_t* mod );
int compass_startup( stg_model_t* mod );
int compass_shutdown( stg_model_t* mod );
int compass_update( stg_model_t* mod );
void compass_load ( stg_model_t *mod );

int compass_render_data( stg_model_t* mod, char* name, void* data, size_t len, void* userp );
int compass_render_cfg( stg_model_t* mod, char* name, void* data, size_t len, void* userp );
int compass_unrender_data( stg_model_t* mod, char* name, void* data, size_t len, void* userp );
int compass_unrender_cfg( stg_model_t* mod, char* name, void* data, size_t len, void* userp );



int compass_init ( stg_model_t* mod ) {
  // override the default methods
  mod->f_startup = compass_startup;
  mod->f_shutdown = compass_shutdown;
  mod->f_update = NULL; // installed at startup/shutdown
  mod->f_load = compass_load;

  // remove the polygon: sensor has no body: 
  stg_model_set_property( mod, "polygons", NULL, 0); 

  stg_geom_t geom;
  memset( &geom, 0, sizeof(geom));
  stg_model_set_property( mod, "geom", &geom, sizeof(geom));  

  stg_color_t color = stg_lookup_color( "magenta" );
  stg_model_set_property( mod, "color", &color, sizeof(color) );
 
  stg_compass_config_t cfg;
  memset(&cfg, 0, sizeof(cfg));
  
  cfg.north = STG_DEFAULT_COMPASS_MAGNORTH;
 
  stg_model_set_property( mod, "compass_cfg", &cfg, sizeof(cfg)); 

  // start with blank data
	stg_compass_t data; 
	memset(&data, 0, sizeof(data)); 

  stg_model_set_property( mod, "compass_data", &data, sizeof(data) );

  // Menu is moved to here? 
  stg_model_add_property_toggles( mod, "compass_data", 
      compass_render_data, 
      NULL, 
      compass_unrender_data,
      NULL, 
      "rangecom data", 
      TRUE ); 

  return 0;
}

void compass_load( stg_model_t* mod )
{
  stg_compass_config_t* now = 
    stg_model_get_property_fixed( mod, "compass_cfg", sizeof(stg_compass_config_t));
  assert(now); 

  stg_compass_config_t cfg;
  memset( &cfg, 0, sizeof(cfg) );
  
  cfg.north = wf_read_angle(mod->id, "compass", now->north );
  
  stg_model_set_property(mod, "compass_cfg", &cfg, sizeof(cfg));
}



int compass_startup( stg_model_t* mod )
{
  PRINT_DEBUG( "compass startup" );  
  
  mod->f_update = compass_update;
  //mod->watts = STG_RANGECOM_WATTS;
  
  return 0;
}

int compass_shutdown( stg_model_t* mod )
{
  mod->f_update = NULL;
  //mod->watts = 0.0;

  // this will unrender the data
  stg_model_set_property( mod, "rangecom_data", NULL, 0 );
  
  return 0;
}

///////////////////////////////////////////////////////////////////////////
// Update the data
int compass_update( stg_model_t* mod )
{
  PRINT_DEBUG( "compass update" );

  if( mod->subs < 1 )
    return 0;

  stg_compass_config_t* cfg = 
    stg_model_get_property_fixed( mod, "compass_cfg", sizeof(stg_compass_config_t) );
  assert(cfg);

  stg_pose_t mypose; 

  stg_model_get_global_pose( mod, &mypose );

  stg_compass_t data;
  memset( &data, 0, sizeof(data)); 


  data.to_north = NORMALIZE(cfg->north - mypose.a); // TODO FIX THIS
  stg_model_set_property( mod, "compass_data",
		  &data, sizeof(stg_compass_t) );
  
  return 0;
}

int compass_unrender_data ( stg_model_t* mod, char* name, void* data, size_t len, void* userp ) {
  stg_model_fig_clear( mod, "compass_data_fig" ); 
  return 1; // cancel callback
} 

int compass_render_data( stg_model_t* mod, char* name, void* data, size_t len, void* userp )
{ 
  stg_rtk_fig_t* fig = stg_model_get_fig( mod, "compass_data_fig" );

  if (!fig) 
    fig = stg_model_fig_create( mod, "compass_data_fig", "top", STG_LAYER_NEIGHBORDATA ); 

	stg_rtk_fig_clear( fig );
  stg_compass_t *direction = (stg_compass_t*) data; 

  stg_rtk_fig_arrow( fig, 0, 0, direction->to_north, 1.0, 0.10 );
  return 0;
}

int compass_render_cfg( stg_model_t* mod, char* name, void* data, size_t len, void* userp)
{
  //????
  return 0;
}

