/*
 *  Player - One Hell of a Robot Server
 *  Copyright (C) 2004  Brian Gerkey gerkey@stanford.edu    
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

/*
 * $Id: mapfile.cc,v 1.12 2005/05/03 00:11:39 gerkey Exp $
 *
 * A driver to read an occupancy grid map from an image file.
 */

/** @addtogroup drivers Drivers */
/** @{ */
/** @defgroup player_driver_mapfile mapfile

The mapfile driver reads a occupancy grid map from a bitmap image file and
provides the map to others via the @ref player_interface_map interface.
Since gdk-pixbuf is used to load the file, pretty much all bitmap formats
are supported.

Each cell in an occupancy grid map takes 1 of 3 states: occupied (1),
unknown (0), and free (-1).  The mapfile driver converts each pixel of an
image to a cell with one of these states in the following way: average
the color values; divide this average by max value to get a ratio; if
this ratio is greater than .95, the cell is occupied; if ratio is less
than 0.1, the cell is free; otherwise it is unknown.  In other words,
"blacker" pixels are occupied, "whiter" pixels are free, and those in
between are unknown.

Note that @ref player_interface_map devices produce no data; the map is
delivered via a sequence of configuration requests.

@par Compile-time dependencies

- gdk-pixbuf-2.0 (usually installed as part of GTK)

@par Provides

- @ref player_interface_map

@par Requires

- None

@par Configuration requests

- PLAYER_MAP_GET_INFO_REQ
- PLAYER_MAP_GET_DATA_REQ

@par Configuration file options

- filename (string)
  - Default: NULL
  - The image file to read.
- resolution (length)
  - Default: -1.0
  - Resolution (length per pixel) of the image.
- negate (integer)
  - Default: 0
  - Should we negate (i.e., invert) the colors in the image before
    reading it?  Useful if you're using the same image file as the
    world bitmap for Stage 1.3.x, which has the opposite semantics for
    free/occupied pixels.
 
@par Example 

@verbatim
driver
(
  name "mapfile"
  provides ["map:0"]
  filename "mymap.pgm"
  resolution 0.1  # 10cm per pixel
)
@endverbatim

@par Authors

Brian Gerkey

*/
/** @} */

#include <sys/types.h> // required by Darwin
#include <netinet/in.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include <player.h>
#include <drivertable.h>
#include <driver.h>
#include <error.h>

// use gdk-pixbuf for image loading
#include <gdk-pixbuf/gdk-pixbuf.h>

// compute linear index for given map coords
#define MAP_IDX(mf, i, j) ((mf->size_x) * (j) + (i))

// check that given coords are valid (i.e., on the map)
#define MAP_VALID(mf, i, j) ((i >= 0) && (i < mf->size_x) && (j >= 0) && (j < mf->size_y))


extern int global_playerport;

class MapFile : public Driver
{
  private:
    const char* filename;
    double resolution;
    int negate;
    int size_x, size_y;
    char* mapdata;
    
    // Handle map info request
    void HandleGetMapInfo(void *client, void *request, int len);
    // Handle map data request
    void HandleGetMapData(void *client, void *request, int len);

  public:
    MapFile(ConfigFile* cf, int section, const char* file, double res, int neg);
    ~MapFile();
    int Setup();
    int Shutdown();
    int PutConfig(player_device_id_t id, void *client, 
                  void* src, size_t len,
                  struct timeval* timestamp);
};

Driver*
MapFile_Init(ConfigFile* cf, int section)
{
  const char* filename;
  double resolution;
  int negate;

  if(!(filename = cf->ReadFilename(section,"filename", NULL)))
  {
    PLAYER_ERROR("must specify map filename");
    return(NULL);
  }
  if((resolution = cf->ReadLength(section,"resolution",-1.0)) < 0)
  {
    PLAYER_ERROR("must specify positive map resolution");
    return(NULL);
  }
  negate = cf->ReadInt(section,"negate",0);

  return((Driver*)(new MapFile(cf, section, filename, resolution, negate)));
}

// a driver registration function
void 
MapFile_Register(DriverTable* table)
{
  table->AddDriver("mapfile", MapFile_Init);
}


// this one has no data or commands, just configs
MapFile::MapFile(ConfigFile* cf, int section, const char* file, double res, int neg) : 
  Driver(cf, section, PLAYER_MAP_CODE, PLAYER_READ_MODE,
         0,0,100,100)
{
  this->mapdata = NULL;
  this->size_x = this->size_y = 0;
  this->filename = file;
  this->resolution = res;
  this->negate = neg;
}

MapFile::~MapFile()
{
}

int
MapFile::Setup()
{
  GdkPixbuf* pixbuf;
  guchar* pixels;
  guchar* p;
  int rowstride, n_channels, bps;
  GError* error = NULL;
  int i,j,k;
  double occ;
  int color_sum;
  double color_avg;

  // Initialize glib
  g_type_init();

  printf("MapFile loading image file: %s...", this->filename);
  fflush(stdout);

  // Read the image
  if(!(pixbuf = gdk_pixbuf_new_from_file(this->filename, &error)))
  {
    PLAYER_ERROR1("failed to open image file %s", this->filename);
    return(-1);
  }

  this->size_x = gdk_pixbuf_get_width(pixbuf);
  this->size_y = gdk_pixbuf_get_height(pixbuf);

  assert(this->mapdata = (char*)malloc(sizeof(char) *
                                       this->size_x * this->size_y));

  rowstride = gdk_pixbuf_get_rowstride(pixbuf);
  bps = gdk_pixbuf_get_bits_per_sample(pixbuf)/8;
  n_channels = gdk_pixbuf_get_n_channels(pixbuf);
  if(gdk_pixbuf_get_has_alpha(pixbuf))
    n_channels++;

  // Read data
  pixels = gdk_pixbuf_get_pixels(pixbuf);
  for(j = 0; j < this->size_y; j++)
  {
    for (i = 0; i < this->size_x; i++)
    {
      p = pixels + j*rowstride + i*n_channels*bps;
      color_sum = 0;
      for(k=0;k<n_channels;k++)
        color_sum += *(p + (k * bps));
      color_avg = color_sum / (double)n_channels;

      if(this->negate)
        occ = color_avg / 255.0;
      else
        occ = (255 - color_avg) / 255.0;
      if(occ > 0.95)
        this->mapdata[MAP_IDX(this,i,this->size_y - j - 1)] = +1;
      else if(occ < 0.1)
        this->mapdata[MAP_IDX(this,i,this->size_y - j - 1)] = -1;
      else
        this->mapdata[MAP_IDX(this,i,this->size_y - j - 1)] = 0;
    }
  }

  gdk_pixbuf_unref(pixbuf);

  puts("Done.");
  printf("MapFile read a %d X %d map, at %.3f m/pix\n",
         this->size_x, this->size_y, this->resolution);
  return(0);
}

int
MapFile::Shutdown()
{
  free(this->mapdata);
  return(0);
}

// Process configuration requests
int 
MapFile::PutConfig(player_device_id_t id, void *client, 
                   void* src, size_t len,
                   struct timeval* timestamp)
{
  // Discard bogus empty packets
  if(len < 1)
  {
    PLAYER_WARN("got zero length configuration request; ignoring");
    if(PutReply(client, PLAYER_MSGTYPE_RESP_NACK,NULL) != 0)
      PLAYER_ERROR("PutReply() failed");
    return(0);
  }

  // Process some of the requests immediately
  switch(((unsigned char*) src)[0])
  {
    case PLAYER_MAP_GET_INFO_REQ:
      HandleGetMapInfo(client, src, len);
      break;
    case PLAYER_MAP_GET_DATA_REQ:
      HandleGetMapData(client, src, len);
      break;
    default:
      PLAYER_ERROR("got unknown config request; ignoring");
      if(PutReply(client, PLAYER_MSGTYPE_RESP_NACK,NULL) != 0)
        PLAYER_ERROR("PutReply() failed");
      break;
  }

  return(0);
}

// Handle map info request
void 
MapFile::HandleGetMapInfo(void *client, void *request, int len)
{
  int reqlen;
  player_map_info_t info;
  
  // Expected length of request
  reqlen = sizeof(info.subtype);
  
  // check if the config request is valid
  if(len != reqlen)
  {
    PLAYER_ERROR2("config request len is invalid (%d != %d)", len, reqlen);
    if (PutReply(client, PLAYER_MSGTYPE_RESP_NACK,NULL) != 0)
      PLAYER_ERROR("PutReply() failed");
    return;
  }

  if(this->mapdata == NULL)
  {
    PLAYER_ERROR("NULL map data");
    if(PutReply(client, PLAYER_MSGTYPE_RESP_NACK,NULL) != 0)
      PLAYER_ERROR("PutReply() failed");
    return;
  }

  // copy in subtype
  info.subtype = ((player_map_info_t*)request)->subtype;
 
  // convert to pixels / kilometer
  info.scale = htonl((uint32_t)rint(1e3 / this->resolution));

  info.width = htonl((uint32_t) (this->size_x));
  info.height = htonl((uint32_t) (this->size_y));

  // Send map info to the client
  if (PutReply(client, PLAYER_MSGTYPE_RESP_ACK, &info, sizeof(info), NULL) != 0)
    PLAYER_ERROR("PutReply() failed");

  return;
}

// Handle map data request
void 
MapFile::HandleGetMapData(void *client, void *request, int len)
{
  int i, j;
  int oi, oj, si, sj;
  int reqlen;
  player_map_data_t data;

  // Expected length of request
  reqlen = sizeof(data) - sizeof(data.data);

  // check if the config request is valid
  if(len != reqlen)
  {
    PLAYER_ERROR2("config request len is invalid (%d != %d)", len, reqlen);
    if(PutReply(client, PLAYER_MSGTYPE_RESP_NACK,NULL) != 0)
      PLAYER_ERROR("PutReply() failed");
    return;
  }

  // Construct reply
  memcpy(&data, request, len);

  oi = ntohl(data.col);
  oj = ntohl(data.row);
  si = ntohl(data.width);
  sj = ntohl(data.height);

  // Grab the pixels from the map
  for(j = 0; j < sj; j++)
  {
    for(i = 0; i < si; i++)
    {
      if((i * j) <= PLAYER_MAP_MAX_CELLS_PER_TILE)
      {
        if(MAP_VALID(this, i + oi, j + oj))
          data.data[i + j * si] = this->mapdata[MAP_IDX(this, i+oi, j+oj)];
        else
        {
          PLAYER_WARN2("requested cell (%d,%d) is offmap", i+oi, j+oj);
          data.data[i + j * si] = 0;
        }
      }
      else
      {
        PLAYER_WARN("requested tile is too large; truncating");
        if(i == 0)
        {
          data.width = htonl(si-1);
          data.height = htonl(j-1);
        }
        else
        {
          data.width = htonl(i);
          data.height = htonl(j);
        }
      }
    }
  }
    
  // Send map info to the client
  if(PutReply(client, PLAYER_MSGTYPE_RESP_ACK, &data, 
              sizeof(data) - sizeof(data.data) + 
              ntohl(data.width) * ntohl(data.height),NULL) != 0)
    PLAYER_ERROR("PutReply() failed");
  return;
}
