/*
 *  Player - One Hell of a Robot Server
 *  Copyright (C) 2000  Brian Gerkey   &  Kasper Stoy
 *                      gerkey@usc.edu    kaspers@robotics.usc.edu
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
///////////////////////////////////////////////////////////////////////////
//
// Desc: Driver for faking localization data from simulation ground truth
// Author: Brian Gerkey
// Date: 4 April 2005
// CVS: $Id: fakelocalize.cc,v 1.3 2005/05/03 19:18:06 gerkey Exp $
//
// Requires - Localize device.
//
///////////////////////////////////////////////////////////////////////////

/** @addtogroup drivers Drivers */
/** @{ */
/** @defgroup player_driver_fakelocalize fakelocalize

The fakelocalize driver polls a simulation device for 2D ground truth pose
data, then reports this data as if it were generated by a localization
system.  This driver is useful for running software (e.g., @ref 
player_util_playernav, @ref player_driver_wavefront) that needs a @ref
player_interface_localize device without incurring the computational
cost of actually running a localization algorithm.

@par Compile-time dependencies

- none

@par Provides

- @ref player_interface_localize

@par Requires

- @ref player_interface_simulation : the device from which to get ground truth

@par Configuration requests

- PLAYER_LOCALIZE_SET_POSE_REQ : acknowledged, but ignored
  
@par Configuration file options

- model (string)
  - Default: NULL
  - Name of simulation model for which we're faking localization.
      
@par Example 

@verbatim
driver
(
  name "stage"
  provides ["6665:simulation:0"]
  plugin "libstage"
  worldfile "foo.world"
)
driver
(
  name "fakelocalize"
  provides ["6665:localize:0"]
  requires ["6665:simulation:0"]
  # a model (probably position) declared in "foo.world"
  model "robot0"
)
@endverbatim

@par Authors

Brian Gerkey

*/
/** @} */

#include <sys/types.h>
#include <netinet/in.h>
#include <unistd.h>

#include "player.h"
#include "error.h"
#include "driver.h"
#include "devicetable.h"
#include "drivertable.h"

#define SLEEPTIME_US 100000

// Driver for computing the free c-space from a laser scan.
class FakeLocalize : public Driver
{
  // Constructor
  public: FakeLocalize( ConfigFile* cf, int section);

  // Setup/shutdown/run routines.
  public: virtual int Setup();
  public: virtual int Shutdown();
  public: virtual void Main();

  // Simulation stuff.
  private: int UpdateData();
  private: void HandleRequests();
  private: Driver *sim_driver;
  private: player_device_id_t sim_id;
  private: const char* model;
};


// Initialization function
Driver* FakeLocalize_Init( ConfigFile* cf, int section)
{
  return ((Driver*) (new FakeLocalize( cf, section)));
}


// a driver registration function
void FakeLocalize_Register(DriverTable* table)
{
  table->AddDriver("fakelocalize", FakeLocalize_Init);
}


////////////////////////////////////////////////////////////////////////////////
// Constructor
FakeLocalize::FakeLocalize( ConfigFile* cf, int section)
    : Driver(cf, section, PLAYER_LOCALIZE_CODE, PLAYER_READ_MODE, 
             sizeof(player_localize_data_t), 0, 5, 5)
{
  // Must have an input sim
  if (cf->ReadDeviceId(&this->sim_id, section, "requires",
                       PLAYER_SIMULATION_CODE, -1, NULL) != 0)
  {
    this->SetError(-1);    
    return;
  }
  this->sim_driver = NULL;

  if(!(this->model = cf->ReadString(section, "model", NULL)))
  {
    PLAYER_ERROR("must specify non-NULL model name");
    this->SetError(-1);
    return;
  }
  if(strlen(this->model) >= PLAYER_SIMULATION_IDENTIFIER_MAXLEN)
  {
    PLAYER_ERROR("model name is too long");
    this->SetError(-1);
    return;
  }

  return;
}


////////////////////////////////////////////////////////////////////////////////
// Set up the device (called by server thread).
int FakeLocalize::Setup()
{
  // Subscribe to the sim.
  this->sim_driver = deviceTable->GetDriver(this->sim_id);
  if(!this->sim_driver)
  {
    PLAYER_ERROR("unable to locate suitable simulation device");
    return(-1);
  }
  if(this->sim_driver->Subscribe(this->sim_id) != 0)
  {
    PLAYER_ERROR("unable to subscribe to simulation device");
    return(-1);
  }
  if(this->UpdateData() < 0)
  {
    PLAYER_ERROR("unable to get pose from simulation device");
    this->sim_driver->Unsubscribe(this->sim_id);
    return(-1);
  }
  this->StartThread();
  return 0;
}


////////////////////////////////////////////////////////////////////////////////
// Shutdown the device (called by server thread).
int FakeLocalize::Shutdown()
{
  // Unsubscribe from devices.
  this->sim_driver->Unsubscribe(this->sim_id);
  
  return 0;
}

int
FakeLocalize::UpdateData()
{
  player_localize_data_t loc_data;
  player_simulation_pose2d_req_t cfg;
  int replen;
  unsigned short reptype;
  struct timeval ts;
  
  // Request pose, don't byteswap, and fill in
  cfg.subtype = PLAYER_SIMULATION_GET_POSE2D;
  strncpy( cfg.name, this->model, PLAYER_SIMULATION_IDENTIFIER_MAXLEN );

  if(((replen = this->sim_driver->Request(this->sim_id, this, 
                                          &cfg, sizeof(cfg.subtype), 
                                          NULL, &reptype, 
                                          &cfg, sizeof(cfg), &ts)) < 
      (int)sizeof(player_simulation_pose2d_req_t)) ||
     (reptype != PLAYER_MSGTYPE_RESP_ACK))
    return(-1);

  // Fill in loc_data, byteswapping as we go.
  loc_data.pending_count = 0;
  loc_data.pending_time_sec = htonl(ts.tv_sec);
  loc_data.pending_time_usec = htonl(ts.tv_usec);
  loc_data.hypoth_count = htonl(1);

  // we didn't byteswap the x,y,a on the way in, so we don't do it on the
  // way out.
  loc_data.hypoths[0].mean[0] = cfg.x;
  loc_data.hypoths[0].mean[1] = cfg.y;
  // convert from degrees to squared arc seconds
  loc_data.hypoths[0].mean[2] = htonl(((int32_t)ntohl(cfg.a))*3600);

  // zero covariance and max weight
  memset(loc_data.hypoths[0].cov,0,sizeof(int64_t)*9);
  loc_data.hypoths[0].alpha = htonl((uint32_t)1e6);

  this->PutData(&loc_data,sizeof(loc_data),&ts);

  return(0);
}

void
FakeLocalize::HandleRequests()
{
  int len;
  void *client;
  char request[PLAYER_MAX_REQREP_SIZE];

  while ((len = GetConfig(this->device_id, &client, 
                          &request, sizeof(request),NULL)) > 0)
  {
    switch (request[0])
    {
      case PLAYER_LOCALIZE_SET_POSE_REQ:
        PutReply(this->device_id,client,PLAYER_MSGTYPE_RESP_ACK,NULL);
        break;
      case PLAYER_LOCALIZE_GET_PARTICLES_REQ:
        PutReply(this->device_id,client,PLAYER_MSGTYPE_RESP_NACK,NULL);
        break;
      default:
        PutReply(this->device_id,client,PLAYER_MSGTYPE_RESP_NACK,NULL);
        break;
    }
  }
}

void
FakeLocalize::Main()
{
  for(;;)
  {
    if(this->UpdateData() < 0)
    {
      PLAYER_ERROR("failed to get pose from simulation device");
      pthread_exit(NULL);
    }
    HandleRequests();
    usleep(SLEEPTIME_US);
  }
}
