/*
 *  Player - One Hell of a Robot Server
 *  Copyright (C) 2000  Brian Gerkey   &  Kasper Stoy
 *                      gerkey@usc.edu    kaspers@robotics.usc.edu
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
/*
 * Desc: Driver for reading log files.
 * Author: Andrew Howard
 * Date: 17 May 2003
 * CVS: $Id: readlog.cc,v 1.37 2006/02/27 22:59:24 gerkey Exp $
 *
 * The writelog driver will write data from another device to a log file.
 * The readlog driver will replay the data as if it can from the real sensors.
 */
/** @ingroup drivers */
/** @{ */
/** @defgroup driver_readlog readlog
 * @brief Playback of logged data

The readlog driver can be used to replay data stored in a log file.
This is particularly useful for debugging client programs, since users
may run their clients against the same data set over and over again.
Suitable log files can be generated using the @ref driver_writelog driver.
The format for the log file can be found in the
@ref tutorial_datalog "data logging tutorial".

See below for an example configuration file; note that the device
id's specified in the provides field must match those stored in the
log file (i.e., data logged as "position2d:0" must also be read back as
"position2d:0").

For help in controlling playback, try @ref util_playervcr.
Note that you must declare a @ref interface_log device to allow
playback control.

@par Compile-time dependencies

- none

@par Provides

The readlog driver can provide the following device interfaces.

- @ref interface_laser
- @ref interface_position2d
- @ref interface_sonar

The following interfaces are supported in principle but are currently
disabled because they need to be updated:

- @ref interface_blobfinder
- @ref interface_camera
- @ref interface_fiducial
- @ref interface_gps
- @ref interface_joystick
- @ref interface_position3d
- @ref interface_wifi

The driver also provides an interface for controlling the playback:

- @ref interface_log

@par Requires

- none

@par Configuration requests

- PLAYER_LOG_SET_READ_STATE_REQ
- PLAYER_LOG_GET_STATE_REQ
- PLAYER_LOG_SET_READ_REWIND_REQ

@par Configuration file options

- filename (filename)
  - Default: NULL
  - The log file to play back.
- speed (float)
  - Default: 1.0
  - Playback speed; 1.0 is real-time
- autoplay (integer)
  - Default: 1
  - Begin playing back log data when first client subscribes
    (as opposed to waiting for the client to tell the @ref
    interface_log device to play).
- autorewind (integer)
  - Default: 0
  - Automatically rewind and play the log file again when the end is
    reached (as opposed to not producing any more data).

@par Example

@verbatim

# Play back odometry and laser data at twice real-time from "mydata.log"
driver
(
  name "readlog"
  filename "mydata.log"
  provides ["position2d:0" "laser:0" "log:0"]
  speed 2.0
)
@endverbatim

@author Andrew Howard

*/
/** @} */

#include <assert.h>
#include <ctype.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <unistd.h>

#include <zlib.h>

#include <libplayercore/playercore.h>

#include "encode.h"
#include "readlog_time.h"

#if 0
// we use this pointer to reset timestamps in the client objects when the
// log gets rewound
#include "clientmanager.h"
extern ClientManager* clientmanager;
#endif

// The logfile driver
class ReadLog: public Driver
{
  // Constructor
  public: ReadLog(ConfigFile* cf, int section);

  // Destructor
  public: ~ReadLog();

  // Initialize the driver
  public: virtual int Setup();

  // Finalize the driver
  public: virtual int Shutdown();

  // Main loop
  public: virtual void Main();

  public: virtual int ProcessMessage(MessageQueue * resp_queue,
                                     player_msghdr_t * hdr,
                                     void * data);
  // Process log interface configuration requests
  private: int ProcessLogConfig(MessageQueue * resp_queue,
                                player_msghdr_t * hdr,
                                void * data);

  // Process position interface configuration requests
  private: int ProcessPositionConfig(MessageQueue * resp_queue,
                                     player_msghdr_t * hdr,
                                     void * data);

  // Process laser interface configuration requests
  private: int ProcessLaserConfig(MessageQueue * resp_queue,
                                  player_msghdr_t * hdr,
                                  void * data);

  // Process sonar interface configuration requests
  private: int ProcessSonarConfig(MessageQueue * resp_queue,
                                  player_msghdr_t * hdr,
                                  void * data);

  // Parse the header info
  private: int ParseHeader(int linenum, int token_count, char **tokens,
                           player_devaddr_t *id, double *dtime,
                           unsigned short* type, unsigned short* subtype);
  // Parse some data
  private: int ParseData(player_devaddr_t id,
                         unsigned short type, unsigned short subtype,
                         int linenum, int token_count, char **tokens,
                         double time);

#if 0
  // Parse blobfinder data
  private: int ParseBlobfinder(player_devaddr_t id,
                               unsigned short type, unsigned short subtype,
                               int linenum,
                               int token_count, char **tokens, double time);

  // Parse camera data
  private: int ParseCamera(player_devaddr_t id,
                           unsigned short type, unsigned short subtype,
                           int linenum,
                          int token_count, char **tokens, double time);

  // Parse fiducial data
  private: int ParseFiducial(player_devaddr_t id,
                             unsigned short type, unsigned short subtype,
                             int linenum,
                          int token_count, char **tokens, double time);

  // Parse gps data
  private: int ParseGps(player_devaddr_t id,
                        unsigned short type, unsigned short subtype,
                        int linenum,
                        int token_count, char **tokens, double time);

  // Parse joystick data
  private: int ParseJoystick(player_devaddr_t id,
                             unsigned short type, unsigned short subtype,
                             int linenum,
                        int token_count, char **tokens, double time);
#endif

  // Parse laser data
  private: int ParseLaser(player_devaddr_t id,
                          unsigned short type, unsigned short subtype,
                          int linenum,
                          int token_count, char **tokens, double time);

  // Parse sonar data
  private: int ParseSonar(player_devaddr_t id,
                          unsigned short type, unsigned short subtype,
                          int linenum,
                          int token_count, char **tokens, double time);
  // Parse position data
  private: int ParsePosition(player_devaddr_t id,
                             unsigned short type, unsigned short subtype,
                             int linenum,
                             int token_count, char **tokens, double time);

#if 0

  // Parse position3d data
  private: int ParsePosition3d(player_devaddr_t id,
                               unsigned short type, unsigned short subtype,
                               int linenum,
                               int token_count, char **tokens, double time);

  // Parse truth data
  private: int ParseTruth(player_devaddr_t id,
                          unsigned short type, unsigned short subtype,
                          int linenum,
                          int token_count, char **tokens, double time);

  // Parse wifi data
  private: int ParseWifi(player_devaddr_t id,
                         unsigned short type, unsigned short subtype,
                         int linenum,
                         int token_count, char **tokens, double time);
#endif

  // List of provided devices
  private: int provide_count;
  private: player_devaddr_t provide_ids[1024];
  // spots to cache metadata for a device (e.g., sonar geometry)
  private: void* provide_metadata[1024];

  // The log interface (at most one of these)
  private: player_devaddr_t log_id;

  // File to read data from
  private: const char *filename;
  private: FILE *file;
  private: gzFile gzfile;

  // Input buffer
  private: size_t line_size;
  private: char *line;

  // File format
  private: char *format;

  // Playback speed (1 = real time, 2 = twice real time)
  private: double speed;

  // Playback enabled?
  public: bool enable;

  // Has a client requested that we rewind?
  public: bool rewind_requested;

  // Should we auto-rewind?  This is set in the log devie in the .cfg
  // file, and defaults to false
  public: bool autorewind;
};


////////////////////////////////////////////////////////////////////////////
// Create a driver for reading log files
Driver* ReadReadLog_Init(ConfigFile* cf, int section)
{
  return ((Driver*) (new ReadLog(cf, section)));
}


////////////////////////////////////////////////////////////////////////////
// Device factory registration
void ReadLog_Register(DriverTable* table)
{
  table->AddDriver("readlog", ReadReadLog_Init);
  return;
}


////////////////////////////////////////////////////////////////////////////
// Constructor
ReadLog::ReadLog(ConfigFile* cf, int section)
    : Driver(cf, section)
{
  int i,j;
  player_devaddr_t id;

  this->filename = cf->ReadFilename(section, "filename", NULL);
  if(!this->filename)
  {
    PLAYER_ERROR("must specify a log file to read from");
    this->SetError(-1);
    return;
  }
  this->speed = cf->ReadFloat(section, "speed", 1.0);

  this->provide_count = 0;
  memset(&this->log_id, 0, sizeof(this->log_id));
  memset(this->provide_metadata,0,sizeof(this->provide_metadata));

  // Get a list of devices to provide
  for (i = 0; i < 1024; i++)
  {
    // TODO: fix the indexing here
    if (cf->ReadDeviceAddr(&id, section, "provides", -1, i, NULL) != 0)
      break;
    if (id.interf == PLAYER_LOG_CODE)
      this->log_id = id;
    else
      this->provide_ids[this->provide_count++] = id;
  }

  // Register the log device
  if (this->log_id.interf == PLAYER_LOG_CODE)
  {
    if (this->AddInterface(this->log_id) != 0)
    {
      this->SetError(-1);
      return;
    }
  }

  // Register all the provides devices
  for (i = 0; i < this->provide_count; i++)
  {
    if (this->AddInterface(this->provide_ids[i]) != 0)
    {
      for(j=0;j<this->provide_count;j++)
      {
        // free any allocated metadata slots
        if(this->provide_metadata[j])
        {
          free(provide_metadata[j]);
          provide_metadata[j] = NULL;
        }
      }
      this->SetError(-1);
      return;
    }

    // if it's sonar, then make a spot to cache geometry info
    if(this->provide_ids[i].interf == PLAYER_SONAR_CODE)
      assert((this->provide_metadata[i] =
              calloc(sizeof(player_sonar_geom_t),1)));
  }

  // Get replay options
  this->enable = cf->ReadInt(section, "autoplay", 1);
  this->autorewind = cf->ReadInt(section, "autorewind", 0);

  // Initialize other stuff
  this->format = strdup("unknown");
  this->file = NULL;
  this->gzfile = NULL;

  // Set up the global time object.  We're just shoving our own in over the
  // pre-existing WallclockTime object.  Not pretty but it works.
  if(GlobalTime)
    delete GlobalTime;
  GlobalTime = new ReadLogTime();

  return;
}


////////////////////////////////////////////////////////////////////////////
// Destructor
ReadLog::~ReadLog()
{
  // Free allocated metadata slots
  for(int i=0;i<this->provide_count;i++)
  {
    if(this->provide_metadata[i])
    {
      free(this->provide_metadata[i]);
      this->provide_metadata[i] = NULL;
    }
  }

  return;
}


////////////////////////////////////////////////////////////////////////////
// Initialize driver
int ReadLog::Setup()
{
  // Reset the time
  ReadLogTime_time.tv_sec = 0;
  ReadLogTime_time.tv_usec = 0;
  ReadLogTime_timeDouble = 0.0;

  // Open the file (possibly compressed)
  if (strlen(this->filename) >= 3 && \
      strcasecmp(this->filename + strlen(this->filename) - 3, ".gz") == 0)
    this->gzfile = gzopen(this->filename, "r");
  else
    this->file = fopen(this->filename, "r");

  if (this->file == NULL)
  {
    PLAYER_ERROR2("unable to open [%s]: %s\n", this->filename, strerror(errno));
    return -1;
  }

  // Rewind not requested by default
  this->rewind_requested = false;

  // Make some space for parsing data from the file.  This size is not
  // an exact upper bound; it's just my best guess.
  this->line_size = PLAYER_MAX_MESSAGE_SIZE;
  this->line = (char*) malloc(this->line_size);
  assert(this->line);

  // Start device thread
  this->StartThread();

  return 0;
}


////////////////////////////////////////////////////////////////////////////
// Finalize the driver
int ReadLog::Shutdown()
{
  // Stop the device thread
  this->StopThread();

  // Free allocated mem
  free(this->line);

  // Close the file
  if (this->gzfile)
    gzclose(this->gzfile);
  else
    fclose(this->file);
  this->gzfile = NULL;
  this->file = NULL;

  return 0;
}


////////////////////////////////////////////////////////////////////////////
// Driver thread
void ReadLog::Main()
{
  int ret;
  int i, len, linenum;
  bool use_stored_tokens;
  int token_count=0;
  char *tokens[4096];
  player_devaddr_t header_id, provide_id;
  struct timeval tv;
  double last_wall_time, curr_wall_time;
  double curr_log_time, last_log_time;
  unsigned short type, subtype;
  bool reading_configs;

  linenum = 0;

  last_wall_time = -1.0;
  last_log_time = -1.0;

  // First thing, we'll read all the configs from the front of the file
  reading_configs = true;
  use_stored_tokens = false;

  while (true)
  {
    pthread_testcancel();

    // Process requests
    if(!reading_configs)
      ProcessMessages();

    // If we're not supposed to playback data, sleep and loop
    if(!this->enable && !reading_configs)
    {
      usleep(10000);
      continue;
    }

    // If a client has requested that we rewind, then do so
    if(!reading_configs && this->rewind_requested)
    {
      // back up to the beginning of the file
      if (this->gzfile)
        ret = gzseek(this->file,0,SEEK_SET);
      else
        ret = fseek(this->file,0,SEEK_SET);

      if(ret < 0)
      {
        // oh well, warn the user and keep going
        PLAYER_WARN1("while rewinding logfile, gzseek()/fseek() failed: %s",
                     strerror(errno));
      }
      else
      {
        linenum = 0;

        // reset the time
        ReadLogTime_time.tv_sec = 0;
        ReadLogTime_time.tv_usec = 0;
        ReadLogTime_timeDouble = 0.0;

#if 0
        // reset time-of-last-write in all clients
        //
        // FIXME: It's not really thread-safe to call this here, because it
        //        writes to a bunch of fields that are also being read and/or
        //        written in the server thread.  But I'll be damned if I'm
        //        going to add a mutex just for this.
        clientmanager->ResetClientTimestamps();
#endif

        // reset the flag
        this->rewind_requested = false;

        PLAYER_MSG0(2, "logfile rewound");
        continue;
      }
    }

    if(!use_stored_tokens)
    {
      // Read a line from the file; note that gzgets is really slow
      // compared to fgets (on uncompressed files), so use the latter.
      if (this->gzfile)
        ret = (gzgets(this->file, this->line, this->line_size) == NULL);
      else
        ret = (fgets(this->line, this->line_size, (FILE*) this->file) == NULL);

      if (ret != 0)
      {
        PLAYER_MSG1(1, "reached end of log file %s", this->filename);
        // File is done, so just loop forever, unless we're on auto-rewind,
        // or until a client requests rewind.
        reading_configs = false;
        while(!this->autorewind && !this->rewind_requested)
        {
          usleep(100000);
          pthread_testcancel();

          // Process requests
          this->ProcessMessages();

          ReadLogTime_timeDouble += 0.1;
          ReadLogTime_time.tv_sec = (time_t)floor(ReadLogTime_timeDouble);
          ReadLogTime_time.tv_sec = (time_t)fmod(ReadLogTime_timeDouble,1.0);
        }

        // request a rewind and start again
        this->rewind_requested = true;
        continue;
      }

      // Possible buffer overflow, so bail
      assert(strlen(this->line) < this->line_size);

      linenum += 1;

      //printf("line %d\n", linenum);
      //continue;

      // Tokenize the line using whitespace separators
      token_count = 0;
      len = strlen(line);
      for (i = 0; i < len; i++)
      {
        if (isspace(line[i]))
          line[i] = 0;
        else if (i == 0)
        {
          assert(token_count < (int) (sizeof(tokens) / sizeof(tokens[i])));
          tokens[token_count++] = line + i;
        }
        else if (line[i - 1] == 0)
        {
          assert(token_count < (int) (sizeof(tokens) / sizeof(tokens[i])));
          tokens[token_count++] = line + i;
        }
      }

      if (token_count >= 1)
      {
        // Discard comments
        if (strcmp(tokens[0], "#") == 0)
          continue;

        // Parse meta-data
        if (strcmp(tokens[0], "##") == 0)
        {
          if (token_count == 4)
          {
            free(this->format);
            this->format = strdup(tokens[3]);
          }
          continue;
        }
      }
    }
    else
      use_stored_tokens = false;

    // Parse out the header info
    if (this->ParseHeader(linenum, token_count, tokens,
                          &header_id, &curr_log_time, &type, &subtype) != 0)
      continue;

    if(reading_configs)
    {
      if(type != PLAYER_MSGTYPE_RESP_ACK)
      {
        // not a config
        reading_configs = false;
        // we'll reuse this tokenized string next time through, instead of
        // reading a fresh line from the file
        use_stored_tokens = true;
        continue;
      }
    }

    // Set the global timestamp
    ::ReadLogTime_timeDouble = curr_log_time;
    ::ReadLogTime_time.tv_sec = (time_t)floor(curr_log_time);
    ::ReadLogTime_time.tv_usec = (time_t)fmod(curr_log_time,1.0);

    gettimeofday(&tv,NULL);
    curr_wall_time = tv.tv_sec + tv.tv_usec/1e6;
    if(!reading_configs)
    {
      // Have we published at least one message from this log?
      if(last_wall_time >= 0)
      {
        // Wait until it's time to publish this message
        while((curr_wall_time - last_wall_time) <
              ((curr_log_time - last_log_time) / this->speed))
        {
          gettimeofday(&tv,NULL);
          curr_wall_time = tv.tv_sec + tv.tv_usec/1e6;
          this->ProcessMessages();
          usleep(1000);
        }
      }

      last_wall_time = curr_wall_time;
      last_log_time = curr_log_time;
    }

    // Look for a matching read interface; data will be output on
    // the corresponding provides interface.
    for (i = 0; i < this->provide_count; i++)
    {
      provide_id = this->provide_ids[i];
      if(Device::MatchDeviceAddress(header_id, provide_id))
        this->ParseData(provide_id, type, subtype,
                        linenum, token_count, tokens, curr_log_time);
    }
  }

  return;
}



////////////////////////////////////////////////////////////////////////////
// Process configuration requests
int ReadLog::ProcessLogConfig(MessageQueue * resp_queue,
                              player_msghdr_t * hdr,
                              void * data)
{
  player_log_set_read_state_t* sreq;
  player_log_get_state_t greq;

  switch(hdr->subtype)
  {
    case PLAYER_LOG_REQ_SET_READ_STATE:
      if(hdr->size != sizeof(player_log_set_read_state_t))
      {
        PLAYER_WARN2("request wrong size (%d != %d)",
                     hdr->size, sizeof(player_log_set_read_state_t));
        return(-1);
      }
      sreq = (player_log_set_read_state_t*)data;
      if(sreq->state)
      {
        puts("ReadLog: start playback");
        this->enable = true;
      }
      else
      {
        puts("ReadLog: stop playback");
        this->enable = false;
      }
      this->Publish(this->log_id, resp_queue,
                    PLAYER_MSGTYPE_RESP_ACK,
                    PLAYER_LOG_REQ_SET_READ_STATE);
      return(0);

    case PLAYER_LOG_REQ_GET_STATE:
      greq.type = PLAYER_LOG_TYPE_READ;
      if(this->enable)
        greq.state = 1;
      else
        greq.state = 0;

      this->Publish(this->log_id, resp_queue,
                    PLAYER_MSGTYPE_RESP_ACK,
                    PLAYER_LOG_REQ_GET_STATE,
                    (void*)&greq, sizeof(greq), NULL);
      return(0);

    case PLAYER_LOG_REQ_SET_READ_REWIND:
      // set the appropriate flag in the manager
      this->rewind_requested = true;

      this->Publish(this->log_id, resp_queue,
                    PLAYER_MSGTYPE_RESP_ACK,
                    PLAYER_LOG_REQ_SET_READ_REWIND);
      return(0);

    default:
      return(-1);
  }
}

int
ReadLog::ProcessPositionConfig(MessageQueue * resp_queue,
                               player_msghdr_t * hdr,
                               void * data)
{
  switch(hdr->subtype)
  {
    case PLAYER_POSITION2D_REQ_GET_GEOM:
      {
        // Find the right place from which to retrieve it
        int j;
        for(j=0;j<this->provide_count;j++)
        {
          if(Device::MatchDeviceAddress(this->provide_ids[j], hdr->addr))
            break;
        }
        if(j>=this->provide_count)
          return(-1);

        if(!this->provide_metadata[j])
          return(-1);

        this->Publish(this->provide_ids[j], resp_queue,
                      PLAYER_MSGTYPE_RESP_ACK, hdr->subtype,
                      this->provide_metadata[j],
                      sizeof(player_position2d_geom_t),
                      NULL);
        return(0);
      }
    default:
      return(-1);
  }
}

int
ReadLog::ProcessLaserConfig(MessageQueue * resp_queue,
                            player_msghdr_t * hdr,
                            void * data)
{
  switch(hdr->subtype)
  {
    case PLAYER_LASER_REQ_GET_GEOM:
      {
        // Find the right place from which to retrieve it
        int j;
        for(j=0;j<this->provide_count;j++)
        {
          if(Device::MatchDeviceAddress(this->provide_ids[j], hdr->addr))
            break;
        }
        if(j>=this->provide_count)
          return(-1);

        if(!this->provide_metadata[j])
          return(-1);

        this->Publish(this->provide_ids[j], resp_queue,
                      PLAYER_MSGTYPE_RESP_ACK, hdr->subtype,
                      this->provide_metadata[j],
                      sizeof(player_laser_geom_t),
                      NULL);
        return(0);
      }
    default:
      return(-1);
  }
}

int
ReadLog::ProcessSonarConfig(MessageQueue * resp_queue,
                            player_msghdr_t * hdr,
                            void * data)
{
  switch(hdr->subtype)
  {
    case PLAYER_SONAR_REQ_GET_GEOM:
      {
        // Find the right place from which to retrieve it
        int j;
        for(j=0;j<this->provide_count;j++)
        {
          if(Device::MatchDeviceAddress(this->provide_ids[j], hdr->addr))
            break;
        }
        if(j>=this->provide_count)
          return(-1);

        if(!this->provide_metadata[j])
          return(-1);

        this->Publish(this->provide_ids[j], resp_queue,
                      PLAYER_MSGTYPE_RESP_ACK, hdr->subtype,
                      this->provide_metadata[j],
                      sizeof(player_sonar_geom_t),
                      NULL);
        return(0);
      }
    default:
      return(-1);
  }
}

int
ReadLog::ProcessMessage(MessageQueue * resp_queue,
                        player_msghdr_t * hdr,
                        void * data)
{
  // Handle log config requests
  if(Message::MatchMessage(hdr, PLAYER_MSGTYPE_REQ, -1,
                           this->log_id))
  {
    return(this->ProcessLogConfig(resp_queue, hdr, data));
  }
  else if((hdr->type == PLAYER_MSGTYPE_REQ) &&
          (hdr->addr.interf == PLAYER_LASER_CODE))
  {
    return(this->ProcessLaserConfig(resp_queue, hdr, data));
  }
  else if((hdr->type == PLAYER_MSGTYPE_REQ) &&
          (hdr->addr.interf == PLAYER_SONAR_CODE))
  {
    return(this->ProcessSonarConfig(resp_queue, hdr, data));
  }
  else if((hdr->type == PLAYER_MSGTYPE_REQ) &&
          (hdr->addr.interf == PLAYER_POSITION2D_CODE))
  {
    return(this->ProcessPositionConfig(resp_queue, hdr, data));
  }
  else
    return -1;
}

////////////////////////////////////////////////////////////////////////////
// Signed int conversion macros
#define NINT16(x) (htons((int16_t)(x)))
#define NUINT16(x) (htons((uint16_t)(x)))
#define NINT32(x) (htonl((int32_t)(x)))
#define NUINT32(x) (htonl((uint32_t)(x)))


////////////////////////////////////////////////////////////////////////////
// Unit conversion macros
#define M_MM(x) ((x) * 1000.0)
#define CM_MM(x) ((x) * 100.0)
#define RAD_DEG(x) ((x) * 180.0 / M_PI)


////////////////////////////////////////////////////////////////////////////
// Parse the header info
int
ReadLog::ParseHeader(int linenum, int token_count, char **tokens,
                     player_devaddr_t *id, double *dtime,
                     unsigned short* type, unsigned short* subtype)
{
  char *name;
  player_interface_t interface;

  if (token_count < 7)
  {
    PLAYER_ERROR2("invalid line at %s:%d", this->filename, linenum);
    return -1;
  }

  name = tokens[3];

  if (lookup_interface(name, &interface) == 0)
  {
    *dtime = atof(tokens[0]);
    id->host = atoi(tokens[1]);
    id->robot = atoi(tokens[2]);
    id->interf = interface.interf;
    id->index = atoi(tokens[4]);
    *type = atoi(tokens[5]);
    *subtype = atoi(tokens[6]);
  }
  else
  {
    PLAYER_WARN1("unknown interface name [%s]", name);
    return -1;
  }

  return 0;
}


////////////////////////////////////////////////////////////////////////////
// Parse data
int ReadLog::ParseData(player_devaddr_t id,
                       unsigned short type, unsigned short subtype,
                       int linenum, int token_count, char **tokens,
                       double time)
{
#if 0
  if (id.interf == PLAYER_BLOBFINDER_CODE)
    return this->ParseBlobfinder(id, type, subtype, linenum,
                                 token_count, tokens, time);
  else if (id.interf == PLAYER_CAMERA_CODE)
    return this->ParseCamera(id, type, subtype, linenum,
                             token_count, tokens, time);
  else if (id.interf == PLAYER_FIDUCIAL_CODE)
    return this->ParseFiducial(id, type, subtype, linenum,
                               token_count, tokens, time);
  else if (id.interf == PLAYER_GPS_CODE)
    return this->ParseGps(id, type, subtype, linenum,
                          token_count, tokens, time);
  else if (id.interf == PLAYER_JOYSTICK_CODE)
    return this->ParseJoystick(id, type, subtype, linenum,
                               token_count, tokens, time);
#endif
  if (id.interf == PLAYER_LASER_CODE)
    return this->ParseLaser(id, type, subtype, linenum,
                            token_count, tokens, time);
  else if (id.interf == PLAYER_SONAR_CODE)
    return this->ParseSonar(id, type, subtype, linenum,
                            token_count, tokens, time);
  else if (id.interf == PLAYER_POSITION2D_CODE)
    return this->ParsePosition(id, type, subtype, linenum,
                               token_count, tokens, time);

#if 0
  else if (id.interf == PLAYER_POSITION3D_CODE)
    return this->ParsePosition3d(id, type, subtype, linenum,
                                 token_count, tokens, time);
  else if (id.interf == PLAYER_TRUTH_CODE)
    return this->ParseTruth(id, type, subtype, linenum,
                            token_count, tokens, time);
  else if (id.interf == PLAYER_WIFI_CODE)
    return this->ParseWifi(id, type, subtype, linenum,
                           token_count, tokens, time);
#endif

  PLAYER_WARN1("unknown interface code [%s]",
               ::lookup_interface_name(0, id.interf));
  return -1;
}

#if 0
////////////////////////////////////////////////////////////////////////////
// Parse blobfinder data
int ReadLog::ParseBlobfinder(player_devaddr_t id, int linenum,
                             int token_count, char **tokens, struct timeval time)
{
  player_blobfinder_data_t data;
  player_blobfinder_blob_t *blob;
  size_t size;
  int i, blob_count;

  if (token_count < 9)
  {
    PLAYER_ERROR2("incomplete line at %s:%d", this->filename, linenum);
    return -1;
  }

  data.width = NUINT16(atoi(tokens[6]));
  data.height = NUINT16(atoi(tokens[7]));
  blob_count = atoi(tokens[8]);
  data.blob_count = NUINT16(blob_count);

  if (token_count < 9 + blob_count * 10)
  {
    PLAYER_ERROR2("incomplete line at %s:%d", this->filename, linenum);
    return -1;
  }

  for (i = 0; i < blob_count; i++)
  {
    blob = data.blobs + i;
    blob->id =  NINT16(atoi(tokens[9 + i]));
    blob->color = NUINT32(atoi(tokens[10 + i]));
    blob->area = NUINT32(atoi(tokens[11 + i]));
    blob->x = NUINT16(atoi(tokens[12 + i]));
    blob->y = NUINT16(atoi(tokens[13 + i]));
    blob->left = NUINT16(atoi(tokens[14 + i]));
    blob->right = NUINT16(atoi(tokens[15 + i]));
    blob->top = NUINT16(atoi(tokens[16 + i]));
    blob->bottom = NUINT16(atoi(tokens[17 + i]));
    blob->range = NUINT16(M_MM(atof(tokens[18 + i])));
  }

  size = sizeof(data) - sizeof(data.blobs) + blob_count * sizeof(data.blobs[0]);
  this->PutMsg(id,NULL,PLAYER_MSGTYPE_DATA,0, &data, size, &time);

  return 0;
}


////////////////////////////////////////////////////////////////////////////
// Parse camera data
int ReadLog::ParseCamera(player_devaddr_t id, int linenum,
                               int token_count, char **tokens, struct timeval time)
{
  player_camera_data_t *data;
  size_t src_size, dst_size;

  if (token_count < 13)
  {
    PLAYER_ERROR2("incomplete line at %s:%d", this->filename, linenum);
    return -1;
  }

  data = (player_camera_data_t*) malloc(sizeof(player_camera_data_t));
  assert(data);

  data->width = NUINT16(atoi(tokens[6]));
  data->height = NUINT16(atoi(tokens[7]));
  data->bpp = atoi(tokens[8]);
  data->format = atoi(tokens[9]);
  data->compression = atoi(tokens[10]);
  data->image_size = NUINT32(atoi(tokens[11]));

  // Check sizes
  src_size = strlen(tokens[12]);
  dst_size = ::DecodeHexSize(src_size);
  assert(dst_size = NUINT32(data->image_size));
  assert(dst_size < sizeof(data->image));

  // Decode string
  ::DecodeHex(data->image, dst_size, tokens[12], src_size);

  this->PutMsg(id,NULL,PLAYER_MSGTYPE_DATA,0, data, sizeof(*data) - sizeof(data->image) + dst_size, &time);

  free(data);

  return 0;
}


////////////////////////////////////////////////////////////////////////////
// Parse fiducial data
int ReadLog::ParseFiducial(player_devaddr_t id, int linenum,
                               int token_count, char **tokens, struct timeval time)
{
  player_fiducial_data_t data;
  int fiducial_count;

  if (token_count < 7)
  {
    PLAYER_ERROR2("incomplete line at %s:%d", this->filename, linenum);
    return -1;
  }

  fiducial_count = atoi( tokens[6] );
  data.count = NUINT16( fiducial_count );

  for( int i = 0; i < fiducial_count; i++ )
  {
    data.fiducials[i].id = NINT16( atof(tokens[13*i + 7]) );
  data.fiducials[i].pos[0] = NINT32(M_MM(atof(tokens[13*i+ 8])));
  data.fiducials[i].pos[1] = NINT32(M_MM(atof(tokens[13*i+ 9])));
  data.fiducials[i].pos[2] = NINT32(M_MM(atof(tokens[13*i+10])));
  data.fiducials[i].rot[0] = NINT32(M_MM(atof(tokens[13*i+11])));
  data.fiducials[i].rot[1] = NINT32(M_MM(atof(tokens[13*i+12])));
  data.fiducials[i].rot[2] = NINT32(M_MM(atof(tokens[13*i+13])));
  data.fiducials[i].upos[0] = NINT32(M_MM(atof(tokens[13*i+14])));
  data.fiducials[i].upos[1] = NINT32(M_MM(atof(tokens[13*i+15])));
  data.fiducials[i].upos[2] = NINT32(M_MM(atof(tokens[13*i+16])));
  data.fiducials[i].urot[0] = NINT32(M_MM(atof(tokens[13*i+17])));
  data.fiducials[i].urot[1] = NINT32(M_MM(atof(tokens[13*i+18])));
  data.fiducials[i].urot[2] = NINT32(M_MM(atof(tokens[13*i+19])));
  }

  this->PutMsg(id,NULL,PLAYER_MSGTYPE_DATA,0, &data, sizeof(data), &time);

  return 0;
}


////////////////////////////////////////////////////////////////////////////
// Parse GPS data
int ReadLog::ParseGps(player_devaddr_t id, int linenum,
                      int token_count, char **tokens, struct timeval time)
{
  player_gps_data_t data;

  if (token_count < 17)
  {
    PLAYER_ERROR2("incomplete line at %s:%d", this->filename, linenum);
    return -1;
  }

  data.time_sec = NUINT32((int) atof(tokens[6]));
  data.time_usec = NUINT32((int) fmod(atof(tokens[6]), 1.0));

  data.latitude = NINT32((int) (60 * 60 * 60 * atof(tokens[7])));
  data.longitude = NINT32((int) (60 * 60 * 60 * atof(tokens[8])));
  data.altitude = NINT32(M_MM(atof(tokens[9])));

  data.utm_e = NINT32(CM_MM(atof(tokens[10])));
  data.utm_n = NINT32(CM_MM(atof(tokens[11])));

  data.hdop = NINT16((int) (10 * atof(tokens[12])));
  data.hdop = NINT16((int) (10 * atof(tokens[13])));
  data.err_horz = NUINT32(M_MM(atof(tokens[14])));
  data.err_vert = NUINT32(M_MM(atof(tokens[15])));

  data.quality = atoi(tokens[16]);
  data.num_sats = atoi(tokens[17]);

  this->PutMsg(id,NULL,PLAYER_MSGTYPE_DATA,0,&data, sizeof(data), &time);

  return 0;
}


////////////////////////////////////////////////////////////////////////////
// Parse joystick data
int ReadLog::ParseJoystick(player_devaddr_t id, int linenum,
                      int token_count, char **tokens, struct timeval time)
{
  player_joystick_data_t data;

  if (token_count < 11)
  {
    PLAYER_ERROR2("incomplete line at %s:%d", this->filename, linenum);
    return -1;
  }

  data.xpos = NINT16((short) atoi(tokens[6]));
  data.ypos = NINT16((short) atoi(tokens[7]));
  data.xscale = NINT16((short) atoi(tokens[8]));
  data.yscale = NINT16((short) atoi(tokens[9]));
  data.buttons = NUINT16((unsigned short) (unsigned int) atoi(tokens[10]));

  this->PutMsg(id,NULL,PLAYER_MSGTYPE_DATA,0, &data, sizeof(data), &time);

  return 0;
}
#endif

////////////////////////////////////////////////////////////////////////////
// Parse laser data
int ReadLog::ParseLaser(player_devaddr_t id,
                        unsigned short type, unsigned short subtype,
                        int linenum,
                        int token_count, char **tokens, double time)
{
  int i, count;

  switch(type)
  {
    case PLAYER_MSGTYPE_DATA:
      switch(subtype)
      {
        case PLAYER_LASER_DATA_SCAN:
          {
            player_laser_data_t data;

            if (token_count < 13)
            {
              PLAYER_ERROR2("incomplete line at %s:%d",
                            this->filename, linenum);
              return -1;
            }

            data.id = atoi(tokens[7]);
            data.min_angle = atof(tokens[8]);
            data.max_angle = atof(tokens[9]);
            data.resolution = atof(tokens[10]);
            data.max_range = atof(tokens[11]);
            data.ranges_count = atoi(tokens[12]);
            data.intensity_count = data.ranges_count;

            count = 0;
            for (i = 13; i < token_count; i += 2)
            {
              data.ranges[count] = atof(tokens[i + 0]);
              data.intensity[count] = atoi(tokens[i + 1]);
              count += 1;
            }

            if (count != (int)data.ranges_count)
            {
              PLAYER_ERROR2("range count mismatch at %s:%d",
                            this->filename, linenum);
              return -1;
            }
            this->Publish(id, NULL, type, subtype,
                          (void*)&data, sizeof(data), &time);
            return(0);
          }

        case PLAYER_LASER_DATA_SCANPOSE:
          {
            player_laser_data_scanpose_t data;

            if (token_count < 16)
            {
              PLAYER_ERROR2("incomplete line at %s:%d",
                            this->filename, linenum);
              return -1;
            }

            data.scan.id = atoi(tokens[7]);
            data.pose.px = atof(tokens[8]);
            data.pose.py = atof(tokens[9]);
            data.pose.pa = atof(tokens[10]);
            data.scan.min_angle = atof(tokens[11]);
            data.scan.max_angle = atof(tokens[12]);
            data.scan.resolution = atof(tokens[13]);
            data.scan.max_range = atof(tokens[14]);
            data.scan.ranges_count = atoi(tokens[15]);
            data.scan.intensity_count = data.scan.ranges_count;

            count = 0;
            for (i = 16; i < token_count; i += 2)
            {
              data.scan.ranges[count] = atof(tokens[i + 0]);
              data.scan.intensity[count] = atoi(tokens[i + 1]);
              count += 1;
            }

            if (count != (int)data.scan.ranges_count)
            {
              PLAYER_ERROR2("range count mismatch at %s:%d",
                            this->filename, linenum);
              return -1;
            }

            this->Publish(id, NULL, type, subtype,
                          (void*)&data, sizeof(data), &time);
            return(0);
          }

        default:
          PLAYER_ERROR1("unknown laser data subtype %d\n", subtype);
          return(-1);
      }
      break;

    case PLAYER_MSGTYPE_RESP_ACK:
      switch(subtype)
      {
        case PLAYER_LASER_REQ_GET_GEOM:
          {
            if(token_count < 12)
            {
              PLAYER_ERROR2("incomplete line at %s:%d",
                            this->filename, linenum);
              return -1;
            }

            // cache it
            player_laser_geom_t* geom =
                    (player_laser_geom_t*)calloc(1,sizeof(player_laser_geom_t));
            assert(geom);

            geom->pose.px = atof(tokens[7]);
            geom->pose.py = atof(tokens[8]);
            geom->pose.pa = atof(tokens[9]);
            geom->size.sl = atof(tokens[10]);
            geom->size.sw = atof(tokens[11]);

            // Find the right place to put it
            int j;
            for(j=0;j<this->provide_count;j++)
            {
              if(Device::MatchDeviceAddress(this->provide_ids[j], id))
                break;
            }
            assert(j<this->provide_count);

            if(this->provide_metadata[j])
              free(this->provide_metadata[j]);

            this->provide_metadata[j] = (void*)geom;

            // nothing to publish
            return(0);
          }

        default:
          PLAYER_ERROR1("unknown laser reply subtype %d\n", subtype);
          return(-1);
      }
      break;

    default:
      PLAYER_ERROR1("unknown laser msg type %d\n", type);
      return(-1);
  }
}

////////////////////////////////////////////////////////////////////////////
// Parse sonar data
int ReadLog::ParseSonar(player_devaddr_t id,
                        unsigned short type, unsigned short subtype,
                        int linenum,
                        int token_count, char **tokens, double time)
{
  switch(type)
  {
    case PLAYER_MSGTYPE_DATA:
      switch(subtype)
      {
        case PLAYER_SONAR_DATA_RANGES:
          {
            player_sonar_data_t data;
            if(token_count < 8)
            {
              PLAYER_ERROR2("invalid line at %s:%d", this->filename, linenum);
              return -1;
            }
            data.ranges_count = atoi(tokens[7]);
            int count = 0;
            for(int i=8;i<token_count;i++)
            {
              data.ranges[count++] = atof(tokens[i]);
            }
            if(count != (int)data.ranges_count)
            {
              PLAYER_ERROR2("range count mismatch at %s:%d",
                            this->filename, linenum);
              return -1;
            }
            this->Publish(id, NULL, type, subtype,
                          (void*)&data, sizeof(data), &time);
            return(0);
          }
        case PLAYER_SONAR_DATA_GEOM:
          {
            player_sonar_geom_t geom;
            if(token_count < 8)
            {
              PLAYER_ERROR2("invalid line at %s:%d", this->filename, linenum);
              return -1;
            }
            geom.poses_count = atoi(tokens[7]);
            int count = 0;
            for(int i=8;i<token_count;i+=3)
            {
              geom.poses[count].px = atof(tokens[i]);
              geom.poses[count].py = atof(tokens[i+1]);
              geom.poses[count].pa = atof(tokens[i+2]);
              count++;
            }
            if(count != (int)geom.poses_count)
            {
              PLAYER_ERROR2("range count mismatch at %s:%d",
                            this->filename, linenum);
              return -1;
            }
            this->Publish(id, NULL, type, subtype,
                          (void*)&geom, sizeof(geom), &time);
            return(0);
          }
        default:
          PLAYER_ERROR1("unknown sonar data subtype %d\n", subtype);
          return(-1);
      }
    case PLAYER_MSGTYPE_RESP_ACK:
      switch(subtype)
      {
        case PLAYER_SONAR_REQ_GET_GEOM:
          {
            if(token_count < 8)
            {
              PLAYER_ERROR2("invalid line at %s:%d", this->filename, linenum);
              return -1;
            }

            // cache it
            player_sonar_geom_t* geom =
                    (player_sonar_geom_t*)calloc(1,sizeof(player_sonar_geom_t));
            assert(geom);

            geom->poses_count = atoi(tokens[7]);
            int count = 0;
            for(int i=8;i<token_count;i+=3)
            {
              geom->poses[count].px = atof(tokens[i]);
              geom->poses[count].py = atof(tokens[i+1]);
              geom->poses[count].pa = atof(tokens[i+2]);
              count++;
            }
            if(count != (int)geom->poses_count)
            {
              PLAYER_ERROR2("range count mismatch at %s:%d",
                            this->filename, linenum);
              free(geom);
              return -1;
            }

            // Find the right place to put it
            int j;
            for(j=0;j<this->provide_count;j++)
            {
              if(Device::MatchDeviceAddress(this->provide_ids[j], id))
                break;
            }
            assert(j<this->provide_count);

            if(this->provide_metadata[j])
              free(this->provide_metadata[j]);

            this->provide_metadata[j] = (void*)geom;

            // nothing to publish
            return(0);
          }
        default:
          PLAYER_ERROR1("unknown sonar reply subtype %d\n", subtype);
          return(-1);
      }
    default:
      PLAYER_ERROR1("unknown sonar message type %d\n", subtype);
      return(-1);
  }
}


////////////////////////////////////////////////////////////////////////////
// Parse position data
int
ReadLog::ParsePosition(player_devaddr_t id,
                       unsigned short type, unsigned short subtype,
                       int linenum,
                       int token_count, char **tokens, double time)
{
  switch(type)
  {
    case PLAYER_MSGTYPE_DATA:
      switch(subtype)
      {
        case PLAYER_POSITION2D_DATA_STATE:
          {
            player_position2d_data_t data;
            if(token_count < 14)
            {
              PLAYER_ERROR2("invalid line at %s:%d", this->filename, linenum);
              return -1;
            }
            data.pos.px = atof(tokens[7]);
            data.pos.py = atof(tokens[8]);
            data.pos.pa = atof(tokens[9]);
            data.vel.px = atof(tokens[10]);
            data.vel.py = atof(tokens[11]);
            data.vel.pa = atof(tokens[12]);
            data.stall = atoi(tokens[13]);

            this->Publish(id, NULL, type, subtype,
                          (void*)&data, sizeof(data), &time);
            return(0);
          }
        default:
          PLAYER_ERROR1("unknown position data subtype %d\n", subtype);
          return(-1);
      }
    case PLAYER_MSGTYPE_RESP_ACK:
      switch(subtype)
      {
        case PLAYER_POSITION2D_REQ_GET_GEOM:
          {
            if(token_count < 12)
            {
              PLAYER_ERROR2("invalid line at %s:%d", this->filename, linenum);
              return -1;
            }

            // cache it
            player_position2d_geom_t* geom =
                    (player_position2d_geom_t*)calloc(1,sizeof(player_position2d_geom_t));
            assert(geom);

            geom->pose.px = atof(tokens[7]);
            geom->pose.py = atof(tokens[8]);
            geom->pose.pa = atof(tokens[9]);
            geom->size.sl = atof(tokens[10]);
            geom->size.sw = atof(tokens[11]);

            // Find the right place to put it
            int j;
            for(j=0;j<this->provide_count;j++)
            {
              if(Device::MatchDeviceAddress(this->provide_ids[j], id))
                break;
            }
            assert(j<this->provide_count);

            if(this->provide_metadata[j])
              free(this->provide_metadata[j]);

            this->provide_metadata[j] = (void*)geom;

            // nothing to publish
            return(0);
          }
        default:
          PLAYER_ERROR1("unknown position reply subtype %d\n", subtype);
          return(-1);
      }
    default:
      PLAYER_ERROR1("unknown position message type %d\n", subtype);
      return(-1);
  }
}

#if 0
////////////////////////////////////////////////////////////////////////////
// Parse position3d data
int ReadLog::ParsePosition3d(player_devaddr_t id, int linenum,
                             int token_count, char **tokens, struct timeval time)
{
 player_position3d_data_t data;

  if (token_count < 19)
  {
    PLAYER_ERROR2("incomplete line at %s:%d", this->filename, linenum);
    return -1;
  }

  data.xpos = NINT32(M_MM(atof(tokens[6])));
  data.ypos = NINT32(M_MM(atof(tokens[7])));
  data.zpos = NINT32(M_MM(atof(tokens[8])));

  data.roll = NINT32(1000 * atof(tokens[9]));
  data.pitch = NINT32(1000 * atof(tokens[10]));
  data.yaw = NINT32(1000 * atof(tokens[11]));

  data.xspeed = NINT32(M_MM(atof(tokens[12])));
  data.yspeed = NINT32(M_MM(atof(tokens[13])));
  data.zspeed = NINT32(M_MM(atof(tokens[14])));

  data.rollspeed = NINT32(1000 * atof(tokens[15]));
  data.pitchspeed = NINT32(1000 * atof(tokens[16]));
  data.yawspeed = NINT32(1000 * atof(tokens[17]));

  data.stall = atoi(tokens[18]);

  this->PutMsg(id,NULL,PLAYER_MSGTYPE_DATA,0, &data, sizeof(data), &time);

  return 0;
}

////////////////////////////////////////////////////////////////////////////
// Parse truth data
int ReadLog::ParseTruth(player_devaddr_t id, int linenum,
                             int token_count, char **tokens, struct timeval time)
{
 player_truth_data_t data;

  if (token_count < 11)
  {
    PLAYER_ERROR2("incomplete line at %s:%d", this->filename, linenum);
    return -1;
  }

  data.pos[0] = NINT32(M_MM(atof(tokens[6])));
  data.pos[1] = NINT32(M_MM(atof(tokens[7])));
  data.pos[2] = NINT32(M_MM(atof(tokens[8])));

  data.rot[0] = NINT32(M_MM(atof(tokens[9])));
  data.rot[1] = NINT32(M_MM(atof(tokens[10])));
  data.rot[2] = NINT32(M_MM(atof(tokens[11])));

  this->PutMsg(id,NULL,PLAYER_MSGTYPE_DATA,0, &data, sizeof(data), &time);

  return 0;
}


////////////////////////////////////////////////////////////////////////////
// Parse wifi data
int ReadLog::ParseWifi(player_devaddr_t id, int linenum,
                              int token_count, char **tokens, struct timeval time)
{
  player_wifi_data_t data;
  player_wifi_link_t *link;
  int i;

  if (token_count < 6)
  {
    PLAYER_ERROR2("incomplete line at %s:%d", this->filename, linenum);
    return -1;
  }

  data.link_count = 0;
  for (i = 6; i < token_count; i += 4)
  {
    link = data.links + data.link_count;

    strcpy(link->ip, tokens[i + 0]);
    link->qual = atoi(tokens[i + 1]);
    link->level = atoi(tokens[i + 2]);
    link->noise = atoi(tokens[i + 3]);

    link->qual = htons(link->qual);
    link->level = htons(link->level);
    link->noise = htons(link->noise);

    data.link_count++;
  }
  data.link_count = htons(data.link_count);

  this->PutMsg(id,NULL,PLAYER_MSGTYPE_DATA,0, &data, sizeof(data), &time);

  return 0;
}
#endif

