/*
 *  Player Java Client - SimulationInterface.java
 *  Copyright (C) 2005 Radu Bogdan Rusu
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: SimulationInterface.java,v 1.4 2005/05/26 09:14:47 veedee Exp $
 *
 */
package javaclient;

import javaclient.structures.PlayerSimulationPose2D;

/**
 * Player devices may either be real hardware or virtual devices generated by a simulator such 
 * as Stage or Gazebo. This interface provides direct access to a simulator.
 * @author Radu Bogdan Rusu
 * @version
 * <ul>
 *      <li>v1.6.3 - Player 1.6.3 (all interfaces) supported
 *      <li>v1.6.2 - Player 1.6.2 supported, Javadoc documentation, several bugfixes  
 * </ul>
 */
public class SimulationInterface extends PlayerDevice {

    private static final boolean isDebugging = PlayerClient.isDebugging;

    /* simulators */
    private final short PLAYER_SIMULATION_CODE = PlayerClient.PLAYER_SIMULATION_CODE;

    /* the player message types (see player.h) */
    private final short PLAYER_MSGTYPE_REQ = PlayerClient.PLAYER_MSGTYPE_REQ;
    
    /* request packet subtypes */ 
    protected final short PLAYER_SIMULATION_SET_POSE2D = 0;
    protected final short PLAYER_SIMULATION_GET_POSE2D = 1;
    
    /** the maximum length of a string indentifying a simulation object */
    public static final short PLAYER_SIMULATION_IDENTIFIER_MAXLEN = 64;
    
    /* object containing player_simulation_pose2d_req in case of a threaded call */
    private PlayerSimulationPose2D psp2d;
    private boolean                readyPSP2D = false;

    /**
     * Constructor for SimulationInterface.
     * @param pc a reference to the PlayerClient object
     * @param indexOfDevice the index of the device
     */
    public SimulationInterface (PlayerClient pc, short indexOfDevice) {
        super(pc);
        device    = PLAYER_SIMULATION_CODE;
        index     = indexOfDevice;
    }
    
    /**
     * Configuration request: set 2D pose of a named simulation object.
     *<br><br>
     * To set or get the pose of an object in a simulator, use this message type. If the 
     * subtype is PLAYER_SIMULATION_SET_POSE2D, the server will ask the simulator to move 
     * the named object to the location specified by (x,y,a) and return ACK.<br><br>
     * See the player_simulation_pose2d_req structure from player.h
     */
    public void set2DPose (String identifier, int x, int y, int theta) {
        try {
            int size = 13 + PLAYER_SIMULATION_IDENTIFIER_MAXLEN;
            sendHeader (PLAYER_MSGTYPE_REQ, size);        /* payload */
            os.writeByte (PLAYER_SIMULATION_SET_POSE2D);
            char[] ident = identifier.toCharArray ();
            for (int i = 0; i < identifier.length (); i++)
                os.writeByte (ident[i]);
            for (int i = 0; i < PLAYER_SIMULATION_IDENTIFIER_MAXLEN-identifier.length (); i++)
                os.writeByte (0);
            os.writeInt (x);
            os.writeInt (y);
            os.writeInt (theta);
            os.flush ();
        } catch (Exception e) {
            System.err.println ("[Simulation] : Couldn't send PLAYER_SIMULATION_SET_POSE2D command: " + 
                    e.toString ());
        }
    }
    
    /**
     * Configuration request: get 2D pose of a named simulation object.
     *<br><br>
     * To set or get the pose of an object in a simulator, use this message type. If the 
     * subtype is PLAYER_SIMULATION_GET_POSE2D, the server will attempt to locate the 
     * named object and reply with the same packet with (x,y,a) filled in. For all message 
     * subtypes, if the named object does not exist, or some other error occurs, the request 
     * should reply NACK.<br><br>
     * See the player_simulation_pose2d_req structure from player.h
     */
    public void get2DPose (String identifier) {
        try {
            int size = 1 + PLAYER_SIMULATION_IDENTIFIER_MAXLEN;
            sendHeader (PLAYER_MSGTYPE_REQ, size);        /* payload */
            os.writeByte (PLAYER_SIMULATION_GET_POSE2D);
            char[] ident = identifier.toCharArray ();
            for (int i = 0; i < identifier.length (); i++)
                os.writeByte (ident[i]);
            for (int i = 0; i < PLAYER_SIMULATION_IDENTIFIER_MAXLEN-identifier.length (); i++)
                os.writeByte (0);
            os.flush ();
        } catch (Exception e) {
            System.err.println ("[Simulation] : Couldn't send PLAYER_SIMULATION_GET_POSE2D command: " + 
                    e.toString ());
        }
    }
    
    /**
     * Handle acknowledgement response messages (threaded mode).
     * @param size size of the payload
     */
    public void handleResponse (int size) {
        if (size == 0) {
            if (isDebugging)
            	System.err.println ("[Simulation][Debug] : Unexpected response of size 0!");
            return;
        }
        try {
            /* each reply begins with a uint8_t subtype field */
            short subtype = is.readByte ();
            switch (subtype) {
                case PLAYER_SIMULATION_GET_POSE2D: {
                    psp2d             = new PlayerSimulationPose2D ();
                    readyPSP2D        = true;
                    
                    /* the identifier of the object we want to locate */
                    String psp2dName  = new String ();
                    for (int j = 0; j < PLAYER_SIMULATION_IDENTIFIER_MAXLEN ; j++)
                        psp2dName += (char)is.readByte ();
                    psp2d.setName (psp2dName);
                    
                    /* the desired pose or returned pose in (mm,mm,degrees) */
                    psp2d.setX     (is.readInt ());
                    psp2d.setY     (is.readInt ());
                    psp2d.setTheta (is.readInt ());
                    break;
                }
                default: {
                    System.err.println ("[Simulation] : Unexpected response " + subtype + 
                            " of size = " + size);
                    break;
                }
            }
        } catch (Exception e) {
            System.err.println ("[Simulation] : Error when reading payload " + e.toString ());
        }
    }

    /**
     * Handle negative acknowledgement response messages.
     */
    public void handleNARMessage () {
        try {
            int size = is.readInt ();    /* read the packet size */
            System.err.println ("[Simulation] : Handling NAR of size = " + size);
        } catch (Exception e) {
            System.err.println ("[Simulation] : handleResponsePosition ERROR " + e.toString ());
        }
    }
    
    /**
     * Get the 2D pose of a named simulation object.
     * @return an object of type PlayerSimulationPose2D containing the required pose data
     */
    public PlayerSimulationPose2D getSimulationPose2D () { return psp2d; }
    
    /**
     * Check if pose data is available.
     * @return true if ready, false if not ready 
     */
    public boolean isPose2DReady () {
        if (readyPSP2D) {
            readyPSP2D = false;
            return true;
        }
        return false;
    }
    
}
